package com.ekoapp.ekosdk.internal.data.boundarycallback

import androidx.arch.core.util.Function
import androidx.core.util.Pair
import com.amity.socialcloud.sdk.chat.data.message.MessageQueryPersister
import com.amity.socialcloud.sdk.chat.data.message.MessageRemoteDataStore
import com.amity.socialcloud.sdk.chat.message.AmityMessage
import com.amity.socialcloud.sdk.core.AmityTags
import com.ekoapp.core.utils.getCurrentClassAndMethodNames
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.internal.api.dto.MessageQueryDto
import com.ekoapp.ekosdk.internal.api.socket.request.MessageQueryRequest.MessageQueryOptions
import com.amity.socialcloud.sdk.log.AmityLog
import com.github.davidmoten.rx2.RetryWhen
import com.github.davidmoten.rx2.RetryWhen.ErrorAndDuration
import com.google.common.collect.Maps
import com.google.common.collect.Sets
import io.reactivex.Completable
import io.reactivex.disposables.Disposable
import io.reactivex.rxjava3.core.Single
import io.reactivex.schedulers.Schedulers
import io.reactivex.subjects.Subject
import java.util.concurrent.Executor
import java.util.concurrent.Executors
import java.util.concurrent.TimeUnit

internal class EkoMessageBoundaryCallback(
    private val channelId: String,
    private val parentId: String?,
    private val isFilterByParentId: Boolean,
    private val isDeleted: Boolean?,
    includingTags: AmityTags,
    excludingTags: AmityTags,
    private val type: AmityMessage.DataType?,
    private val stackFromEnd: Boolean, pageSize: Int,
    private val delaySubject: Subject<Boolean>
)

    : EkoMessageWithTagBoundaryCallback(includingTags, excludingTags, pageSize),
    Function<AmityMessage, AmityMessage> {
    
    private val TAG = javaClass.name

    private val messageIdAndTokenMap: MutableMap<String, Pair<String, Boolean>> =
        Maps.newConcurrentMap()
    private val messageIdSet = Sets.newConcurrentHashSet<String>()

    override fun onSubscribe(d: Disposable) {}
    override fun onComplete() {}
    override fun onError(e: Throwable) {
        val TAG = getCurrentClassAndMethodNames()
        AmityLog.tag(TAG).e(e)
    }

    fun onFirstLoaded() {
        val options = MessageQueryOptions()
        if (stackFromEnd) {
            options.last = pageSize
        } else {
            options.first = pageSize
        }
        AmityLog.tag(TAG).i("onFirstLoaded")
        call(options)
            .doOnComplete { delaySubject.onComplete() }
            .doOnError { delaySubject.onComplete() }
            .subscribeOn(Schedulers.from(SINGLE_THREAD_EXECUTOR))
            .subscribe(this)
    }

    override fun apply(input: AmityMessage): AmityMessage {
        messageIdSet.add(input.getMessageId())
        mapByMessage(input.getMessageId())
        return input
    }

    private fun mapByMessage(messageId: String) {
        messageIdAndTokenMap[messageId]?.let { messageIdAndTokenMapNonNull ->
            if (messageIdAndTokenMapNonNull.first.isNullOrEmpty() || messageIdAndTokenMapNonNull.second == true) {
                return
            }
            val options = MessageQueryOptions()
            options.token = messageIdAndTokenMapNonNull.first
            call(options)
                .doOnSubscribe {
                    messageIdAndTokenMap[messageId] = Pair(messageIdAndTokenMapNonNull.first, true)
                }
                .doOnError {
                    messageIdAndTokenMap[messageId] = Pair(messageIdAndTokenMapNonNull.first, false)
                }
                .subscribeOn(Schedulers.from(SINGLE_THREAD_EXECUTOR))
                .subscribe(this)
        }
    }

    private fun call(options: MessageQueryOptions?): Completable {
        return MessageRemoteDataStore().queryMessages(
            channelId = channelId,
            filterByParentId = isFilterByParentId,
            parentId = parentId,
            isDeleted = isDeleted,
            includingTags = includingTags,
            excludingTags = excludingTags,
            dataType = type?.apiKey,
            first = options?.first,
            last = options?.last,
            token = options?.token
        ).toV2()
            .flatMap {
                MessageQueryPersister().persist(it)
                    .andThen(Single.just(it)).toV2()
            }
            .doOnSuccess { dto: MessageQueryDto ->
                val messages = dto.messages
                if (!messages.isNullOrEmpty()) {
                    val messageId =
                        if (stackFromEnd) messages.firstOrNull() { if (this.isDeleted == null) true else this.isDeleted == it.isDeleted }?.messageId
                        else messages.lastOrNull { if (this.isDeleted == null) true else this.isDeleted == it.isDeleted }?.messageId
                    val nextPage = if (stackFromEnd) dto.token.previous
                    else dto.token.next
                    if (!nextPage.isNullOrEmpty()) {
                        if (!messageId.isNullOrEmpty()) {
                            messageIdAndTokenMap[messageId] = Pair.create(nextPage, false)
                            if (messageIdSet.contains(messageId)) {
                                mapByMessage(messageId)
                            }
                        } else {
                            val options = MessageQueryOptions()
                            options.token = nextPage
                            call(options)
                                .subscribeOn(Schedulers.from(SINGLE_THREAD_EXECUTOR))
                                .subscribe(this)
                        }
                    }
                }
            }
            .retryWhen(RetryWhen.maxRetries(3)
                .exponentialBackoff(1, 10, TimeUnit.SECONDS, 1.5)
                .action { errorAndDuration: ErrorAndDuration ->
                    AmityLog.tag(TAG).e(
                        errorAndDuration.throwable(),
                        "an error occurred, back-off for durationMs:%s",
                        errorAndDuration.durationMs()
                    )
                }
                .build())
            .ignoreElement()
    }

    companion object {
        private val SINGLE_THREAD_EXECUTOR: Executor = Executors.newSingleThreadExecutor()
    }

    init {
        onFirstLoaded()
    }

}
