package com.ekoapp.ekosdk.internal.data.boundarycallback

import androidx.arch.core.util.Function
import androidx.core.util.Pair
import androidx.paging.PagedList
import com.amity.socialcloud.sdk.video.stream.AmityStream
import com.ekoapp.ekosdk.internal.api.EkoSocket
import com.ekoapp.ekosdk.internal.api.dto.EkoStreamQueryDto
import com.ekoapp.ekosdk.internal.api.socket.call.Call
import com.ekoapp.ekosdk.internal.api.socket.call.StreamQueryConverter
import com.ekoapp.ekosdk.internal.api.socket.request.StreamQueryRequest
import com.ekoapp.ekosdk.internal.api.socket.request.StreamQueryRequest.StreamQueryOptions
import com.amity.socialcloud.sdk.log.AmityLog
import com.github.davidmoten.rx2.RetryWhen
import com.github.davidmoten.rx2.RetryWhen.ErrorAndDuration
import com.google.common.collect.Maps
import com.google.common.collect.Sets
import io.reactivex.Completable
import io.reactivex.CompletableObserver
import io.reactivex.disposables.Disposable
import io.reactivex.schedulers.Schedulers
import io.reactivex.subjects.Subject
import java.util.concurrent.Executor
import java.util.concurrent.Executors
import java.util.concurrent.TimeUnit

internal class EkoStreamBoundaryCallback(val statuses: Array<String>,
                                         val isReconnecting: Boolean,
                                         val pageSize: Int,
                                         private val delaySubject: Subject<Boolean>) : PagedList.BoundaryCallback<AmityStream>(), CompletableObserver, Function<AmityStream, AmityStream> {

    companion object {
        private val SINGLE_THREAD_EXECUTOR: Executor = Executors.newSingleThreadExecutor()
    }
    
    private val TAG = javaClass.name

    private val streamIdAndTokenMap: MutableMap<String, Pair<String?, Boolean>> = Maps.newConcurrentMap()
    private val streamIdSet = Sets.newConcurrentHashSet<String>()

    init {
        onFirstLoaded()
    }

    override fun onSubscribe(d: Disposable) = Unit

    override fun onComplete() = Unit

    override fun onError(e: Throwable) = Unit

    fun onFirstLoaded() {
        val options = StreamQueryOptions(limit = pageSize)
        call(options)
                .doOnComplete { delaySubject.onComplete() }
                .doOnError { delaySubject.onComplete() }
                .subscribeOn(Schedulers.from(SINGLE_THREAD_EXECUTOR))
                .subscribe(this)
    }

    override fun apply(input: AmityStream): AmityStream {
        streamIdSet.add(input.getStreamId())
        mapByStream(input.getStreamId())
        return input
    }

    private fun mapByStream(userId: String) {
        streamIdAndTokenMap[userId]?.let { tokenAndStatusNonNull ->
            if (tokenAndStatusNonNull.first.isNullOrEmpty() || tokenAndStatusNonNull.second == true) {
                return
            }
            val options = StreamQueryOptions()
            options.token = tokenAndStatusNonNull.first
            AmityLog.tag(TAG).i("map userId:$userId")
            call(options)
                    .doOnSubscribe { streamIdAndTokenMap[userId] = Pair(tokenAndStatusNonNull.first, true) }
                    .doOnError { streamIdAndTokenMap[userId] = Pair(tokenAndStatusNonNull.first, false) }
                    .subscribeOn(Schedulers.from(SINGLE_THREAD_EXECUTOR))
                    .subscribe(this)
        }
    }

    private fun call(options: StreamQueryOptions): Completable {
        val request = StreamQueryRequest(statuses = statuses.toList(), isDeleted = false, options = options)

        return EkoSocket.call(Call.create(request, StreamQueryConverter()))
                .doOnSuccess { dto: EkoStreamQueryDto ->
                    val streams = dto.result.streams
                    if (!streams.isNullOrEmpty()) {
                        val stream = streams[streams.size - 1]
                        streamIdAndTokenMap[stream.streamId] = Pair.create(dto.token.next ?: "", false)
                        if (streamIdSet.contains(stream.streamId)) {
                            mapByStream(stream.streamId)
                        }
                    }
                }
                .retryWhen(RetryWhen.maxRetries(3)
                        .exponentialBackoff(1, 10, TimeUnit.SECONDS, 1.5)
                        .action { errorAndDuration: ErrorAndDuration -> AmityLog.tag(TAG).e(errorAndDuration.throwable(), "an error occurred, back-off for durationMs:%s", errorAndDuration.durationMs()) }
                        .build())
                .ignoreElement()
    }
}