package com.ekoapp.ekosdk.internal.data.boundarycallback

import androidx.paging.PagedList
import com.amity.socialcloud.sdk.AmityCoreClient
import com.amity.socialcloud.sdk.core.user.AmityFollowRelationship
import com.amity.socialcloud.sdk.core.user.AmityFollowStatusFilter
import com.ekoapp.ekosdk.internal.api.EkoSocket
import com.ekoapp.ekosdk.internal.api.socket.call.Call
import com.ekoapp.ekosdk.internal.api.socket.call.FollowQueryConverter
import com.ekoapp.ekosdk.internal.api.socket.request.FollowerQueryRequest
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.dao.FollowerQueryTokenDao
import com.ekoapp.ekosdk.internal.data.model.FollowerQueryToken
import io.reactivex.Completable
import io.reactivex.schedulers.Schedulers

// unix page number for backward compat!
private const val PAGE_NUMBER = 1

internal class FollowersBoundaryCallback(
        private val userId: String,
        private val filter: AmityFollowStatusFilter,
        val pageSize: Int
) : PagedList.BoundaryCallback<AmityFollowRelationship>() {

    init {
        onFirstLoaded()
    }

    private val tokenDao: FollowerQueryTokenDao = UserDatabase.get().followerQueryTokenDao()

    private fun onFirstLoaded() {
        val options = FollowerQueryRequest.FollowQueryOptions(limit = pageSize)
        val request = FollowerQueryRequest(
                userId = getUserIdParam(userId),
                status = filter.apiKey,
                options = options
        )
        call(request)
                .subscribeOn(Schedulers.io())
                .subscribe()
    }

    override fun onItemAtEndLoaded(itemAtEnd: AmityFollowRelationship) {
        tokenDao.getQueryTokenByPageNumber(
                primaryKeys = mapOf(
                        "userId" to userId,
                        "filter" to filter.apiKey
                ), pageNumber = PAGE_NUMBER
        )
                .filter { token: FollowerQueryToken -> !token.next.isNullOrEmpty() }
                .flatMapCompletable { token: FollowerQueryToken ->
                    call(
                            FollowerQueryRequest(
                                    userId = getUserIdParam(userId),
                                    status = filter.apiKey,
                                    options = FollowerQueryRequest.FollowQueryOptions(token = token.next!!)
                            )
                    )
                }
                .subscribeOn(Schedulers.io())
                .subscribe()
    }

    private fun call(request: FollowerQueryRequest): Completable {
        return EkoSocket.call(Call.create(request, FollowQueryConverter()))
                .doOnSuccess { dto ->
                    val token = FollowerQueryToken().apply {
                        this.userId = this@FollowersBoundaryCallback.userId
                        this.filter = this@FollowersBoundaryCallback.filter
                        this.pageNumber = PAGE_NUMBER
                        this.next = dto.paging?.next
                        this.previous = dto.paging?.previous
                        this.ids = dto.follows?.map { it.getId() }?.toList() ?: listOf()
                        this.updatedAt = AmityCoreClient.getServerTime()
                    }
                    val isFirstPage = dto.paging?.previous == null
                    if (isFirstPage) {
                        tokenDao.insertToken(token)
                                .subscribeOn(Schedulers.io())
                                .subscribe()
                    } else {
                        updateQueryToken(token)
                    }
                }
                .ignoreElement()
    }

    private fun updateQueryToken(token: FollowerQueryToken) {
        tokenDao.getQueryTokenByPageNumber(
                primaryKeys = mapOf(
                        "userId" to userId,
                        "filter" to filter.apiKey
                ), pageNumber = PAGE_NUMBER
        ).flatMapCompletable {
            token.ids.addAll(0, it.ids)
            tokenDao.insertToken(token)
        }.subscribeOn(Schedulers.io())
                .subscribe()
    }

    private fun getUserIdParam(userId: String) : String? {
        return if(userId == AmityCoreClient.getUserId()) null else userId
    }

}