package com.ekoapp.ekosdk.internal.data.boundarycallback

import androidx.paging.PagedList
import com.amity.socialcloud.sdk.AmityCoreClient
import com.amity.socialcloud.sdk.social.feed.AmityPost
import com.ekoapp.ekosdk.internal.api.EkoSocket
import com.ekoapp.ekosdk.internal.api.socket.call.Call
import com.ekoapp.ekosdk.internal.api.socket.call.GlobalPostQueryConverter
import com.ekoapp.ekosdk.internal.api.socket.request.FeedQueryOptions
import com.ekoapp.ekosdk.internal.api.socket.request.GlobalFeedQueryRequest
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.dao.GlobalFeedQueryTokenDao
import com.ekoapp.ekosdk.internal.data.model.GlobalFeedQueryTokenEntity
import io.reactivex.Completable
import io.reactivex.schedulers.Schedulers

// unix page number for backward compat!
private const val PAGE_NUMBER = 1

internal class GlobalFeedBoundaryCallback(
        val pageSize: Int
) : PagedList.BoundaryCallback<AmityPost>() {

    init {
        onFirstLoaded()
    }

    private val tokenDao: GlobalFeedQueryTokenDao = UserDatabase.get().globalFeedQueryTokenDao()

    private fun onFirstLoaded() {
        val request = GlobalFeedQueryRequest(
                options = FeedQueryOptions(limit = pageSize)
        )
        call(request)
                .subscribeOn(Schedulers.io())
                .subscribe()
    }

    override fun onItemAtEndLoaded(itemAtEnd: AmityPost) {
        tokenDao.getQueryTokenByPageNumber(
                primaryKeys = mapOf(
                        "isDeleted" to false
                ), pageNumber = PAGE_NUMBER
        )
                .filter { token: GlobalFeedQueryTokenEntity -> !token.next.isNullOrEmpty() }
                .flatMapCompletable { token: GlobalFeedQueryTokenEntity ->
                    call(
                            GlobalFeedQueryRequest(
                                    options = FeedQueryOptions(token = token.next!!)
                            )
                    )
                }
                .subscribeOn(Schedulers.io())
                .subscribe()
    }

    private fun call(request: GlobalFeedQueryRequest): Completable {
        return EkoSocket.call(Call.create(request, GlobalPostQueryConverter()))
                .doOnSuccess { dto ->
                    val token = GlobalFeedQueryTokenEntity().apply {
                        this.pageNumber = PAGE_NUMBER
                        this.next = dto.paging?.next
                        this.previous = dto.paging.previous
                        this.ids = dto.posts?.map { it.postId }
                        this.updatedAt = AmityCoreClient.getServerTime()
                    }
                    val isFirstPage = dto.paging?.previous == null
                    if (isFirstPage) {
                        tokenDao.insertToken(token)
                                .subscribeOn(Schedulers.io())
                                .subscribe()
                    } else {
                        updateQueryToken(token)
                    }
                }
                .ignoreElement()
    }

    private fun updateQueryToken(token: GlobalFeedQueryTokenEntity) {
        tokenDao.getQueryTokenByPageNumber(
                primaryKeys = primaryKeys(),
            pageNumber = PAGE_NUMBER
        ).flatMapCompletable {
            val ids = arrayListOf<String>()
            ids.addAll(it.ids)
            ids.addAll(token.ids)
            token.ids = ids
            tokenDao.insertToken(token)
        }.subscribeOn(Schedulers.io())
                .subscribe()
    }

    private fun primaryKeys() : Map<String,Any> {
        return mapOf(
            "isDeleted" to false
        )
    }

}