package com.ekoapp.ekosdk.internal.data.boundarycallback

import androidx.paging.PagedList
import com.amity.socialcloud.sdk.AmityCoreClient
import com.amity.socialcloud.sdk.core.user.AmityUser
import com.ekoapp.core.utils.requiredNotEmpty
import com.ekoapp.ekosdk.internal.api.EkoSocket
import com.ekoapp.ekosdk.internal.api.socket.call.Call
import com.ekoapp.ekosdk.internal.api.socket.call.UserQueryConverter
import com.ekoapp.ekosdk.internal.api.socket.request.UserQueryRequest
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.dao.UserQueryTokenDao
import com.ekoapp.ekosdk.internal.data.model.UserQueryTokenEntity
import io.reactivex.Completable
import io.reactivex.schedulers.Schedulers

// unix page number for backward compat!
private const val PAGE_NUMBER = 1

internal class UserBoundaryCallback(
        private val keyword: String,
        private val sortOption: String,
        val pageSize: Int
) : PagedList.BoundaryCallback<AmityUser>() {

    init {
        onFirstLoaded()
    }

    private val tokenDao: UserQueryTokenDao = UserDatabase.get().userQueryTokenDao()

    private fun onFirstLoaded() {
        val request = UserQueryRequest(
            keyword = requiredNotEmpty(keyword),
            sortBy = sortOption,
            options = UserQueryRequest.UserQueryOptions(limit = pageSize)
        )
        call(request)
                .subscribeOn(Schedulers.io())
                .subscribe()
    }

    override fun onItemAtEndLoaded(itemAtEnd: AmityUser) {
        tokenDao.getQueryTokenByPageNumber(
                primaryKeys = mapOf(
                        "keyword" to keyword,
                        "sortBy" to sortOption
                ), pageNumber = PAGE_NUMBER
        )
                .filter { token: UserQueryTokenEntity -> !token.next.isNullOrEmpty() }
                .flatMapCompletable { token: UserQueryTokenEntity ->
                    call(
                            UserQueryRequest(
                                    keyword = requiredNotEmpty(keyword),
                                    sortBy = sortOption,
                                    options = UserQueryRequest.UserQueryOptions(token = token.next!!)
                            )
                    )
                }
                .subscribeOn(Schedulers.io())
                .subscribe()
    }

    private fun call(request: UserQueryRequest): Completable {
        return EkoSocket.call(Call.create(request, UserQueryConverter()))
                .doOnSuccess { dto ->
                    val token = UserQueryTokenEntity().apply {
                        this.keyword = this@UserBoundaryCallback.keyword
                        this.sortBy = this@UserBoundaryCallback.sortOption
                        this.pageNumber = PAGE_NUMBER
                        this.next = dto.token?.next
                        this.previous = dto.token?.previous
                        this.ids = dto.users?.map { it.userId }
                        this.updatedAt = AmityCoreClient.getServerTime()
                    }
                    val isFirstPage = dto.token?.previous == null
                    if (isFirstPage) {
                        tokenDao.insertToken(token)
                                .subscribeOn(Schedulers.io())
                                .subscribe()
                    } else {
                        updateQueryToken(token)
                    }
                }
                .ignoreElement()
    }

    private fun updateQueryToken(token: UserQueryTokenEntity) {
        tokenDao.getQueryTokenByPageNumber(
                primaryKeys = mapOf(
                        "keyword" to keyword,
                        "sortBy" to sortOption
                ), pageNumber = PAGE_NUMBER
        ).flatMapCompletable {
            token.ids.addAll(0, it.ids)
            tokenDao.insertToken(token)
        }.subscribeOn(Schedulers.io())
                .subscribe()
    }

}
