package com.ekoapp.ekosdk.internal.data.boundarycallback

import androidx.paging.PagedList
import com.amity.socialcloud.sdk.AmityCoreClient
import com.amity.socialcloud.sdk.social.feed.AmityPost
import com.ekoapp.ekosdk.internal.api.EkoSocket
import com.ekoapp.ekosdk.internal.api.socket.call.Call
import com.ekoapp.ekosdk.internal.api.socket.call.PostQueryConverter
import com.ekoapp.ekosdk.internal.api.socket.request.FeedQueryOptions
import com.ekoapp.ekosdk.internal.api.socket.request.UserFeedQueryRequest
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.dao.UserFeedQueryTokenDao
import com.ekoapp.ekosdk.internal.data.model.UserFeedQueryTokenEntity
import io.reactivex.Completable
import io.reactivex.schedulers.Schedulers

// unix page number for backward compat!
private const val PAGE_NUMBER = 1

internal class UserFeedBoundaryCallback(
        private val userId: String,
        private val sortOption: String,
        private val isDeleted: Boolean?,
        val pageSize: Int
) : PagedList.BoundaryCallback<AmityPost>() {

    init {
        onFirstLoaded()
    }

    private val tokenDao: UserFeedQueryTokenDao = UserDatabase.get().userFeedQueryTokenDao()

    private fun onFirstLoaded() {
        val request = UserFeedQueryRequest(
                userId = userId,
                sortBy = sortOption,
                isDeleted = isDeleted,
                options = FeedQueryOptions(limit = pageSize)
        )
        call(request)
                .subscribeOn(Schedulers.io())
                .subscribe()
    }

    override fun onItemAtEndLoaded(itemAtEnd: AmityPost) {
        tokenDao.getQueryTokenByPageNumber(
                primaryKeys = mapOf(
                        "userId" to userId,
                        "sortBy" to sortOption,
                        "isDeleted" to (isDeleted?.toString() ?: "")
                ), pageNumber = PAGE_NUMBER
        )
                .filter { token: UserFeedQueryTokenEntity -> !token.next.isNullOrEmpty() }
                .flatMapCompletable { token: UserFeedQueryTokenEntity ->
                    call(
                            UserFeedQueryRequest(
                                    userId = userId,
                                    sortBy = sortOption,
                                    isDeleted = isDeleted,
                                    options = FeedQueryOptions(token = token.next!!)
                            )
                    )
                }
                .subscribeOn(Schedulers.io())
                .subscribe()
    }

    private fun call(request: UserFeedQueryRequest): Completable {
        return EkoSocket.call(Call.create(request, PostQueryConverter()))
                .doOnSuccess { dto ->
                    val token = UserFeedQueryTokenEntity().apply {
                        this.userId = this@UserFeedBoundaryCallback.userId
                        this.isDeleted = this@UserFeedBoundaryCallback.isDeleted?.toString() ?: ""
                        this.sortBy = this@UserFeedBoundaryCallback.sortOption
                        this.pageNumber = PAGE_NUMBER
                        this.next = dto.paging?.next
                        this.previous = dto.paging.previous
                        this.ids = dto.posts?.map { it.postId }
                        this.updatedAt = AmityCoreClient.getServerTime()
                    }
                    val isFirstPage = dto.paging?.previous == null
                    if (isFirstPage) {
                        tokenDao.insertToken(token)
                                .subscribeOn(Schedulers.io())
                                .subscribe()
                    } else {
                        updateQueryToken(token)
                    }
                }
                .ignoreElement()
    }

    private fun updateQueryToken(token: UserFeedQueryTokenEntity) {
        tokenDao.getQueryTokenByPageNumber(
                primaryKeys = mapOf(
                        "userId" to userId,
                        "sortBy" to sortOption,
                        "isDeleted" to (isDeleted?.toString() ?: "")
                ), pageNumber = PAGE_NUMBER
        ).flatMapCompletable {
            token.ids.addAll(0, it.ids)
            tokenDao.insertToken(token)
        }.subscribeOn(Schedulers.io())
                .subscribe()
    }

}