package com.ekoapp.ekosdk.internal.data.dao;

import androidx.annotation.NonNull;
import androidx.paging.DataSource;
import androidx.room.Dao;
import androidx.room.Query;
import androidx.room.Transaction;

import com.amity.socialcloud.sdk.AmityCoreClient;
import com.amity.socialcloud.sdk.core.AmityTags;
import com.amity.socialcloud.sdk.core.user.AmityMembershipType;
import com.ekoapp.ekosdk.EkoChannelReadStatus;
import com.ekoapp.ekosdk.EkoChannelWithMembershipAndExtra;
import com.ekoapp.ekosdk.internal.EkoChannelEntity;
import com.ekoapp.ekosdk.internal.data.UserDatabase;
import com.ekoapp.ekosdk.internal.data.model.EkoChannelTag;
import com.google.common.base.Function;
import com.google.common.collect.FluentIterable;

import org.checkerframework.checker.nullness.compatqual.NullableDecl;
import org.joda.time.DateTime;

import java.util.List;

import io.reactivex.rxjava3.core.Flowable;


@Dao
public abstract class EkoChannelDao extends EkoObjectDao<EkoChannelEntity> implements AmityPagingDao<EkoChannelWithMembershipAndExtra> {

    private final EkoChannelMembershipDao channelMembershipDao;
    private final EkoMessageDao messageDao;
    private final EkoChannelTagDao channelTagDao;
    private final EkoChannelExtraDao channelExtraDao;

    EkoChannelDao() {
        UserDatabase db = UserDatabase.get();
        channelMembershipDao = db.channelMembershipDao();
        messageDao = db.messageDao();
        channelTagDao = db.channelTagDao();
        channelExtraDao = db.channelExtraDao();
    }

    @Query("SELECT channel.*," +
            " channel_membership.readToSegment as membership_readToSegment," +
            " channel_extra.localReadToSegment as extra_localReadToSegment," +
            " channel_membership.lastMentionedSegment as membership_lastMentionedSegment," +
            " channel_extra.localLastMentionedSegment as extra_localLastMentionedSegment" +
            " from channel, channel_membership, channel_extra" +
            " where channel.channelId = channel_membership.channelId and channel.channelId = channel_extra.channelId" +
            " and channel.channelId in (SELECT channelId from channel_membership where membership = :membership)" +
            " and channel_membership.userId = :userId")
    abstract Flowable<List<EkoChannelWithMembershipAndExtra>> getAllJoinedChannelsImpl(String userId, String membership);

    public Flowable<List<EkoChannelWithMembershipAndExtra>> getAllJoinedChannels(String userId) {
        return getAllJoinedChannelsImpl(userId, AmityMembershipType.MEMBER.getApiKey());
    }

    @Query("SELECT channel.*," +
            " channel_membership.readToSegment as membership_readToSegment," +
            " channel_extra.localReadToSegment as extra_localReadToSegment," +
            " channel_membership.lastMentionedSegment as membership_lastMentionedSegment," +
            " channel_extra.localLastMentionedSegment as extra_localLastMentionedSegment" +
            " from channel, channel_membership, channel_extra" +
            " where channel.channelId = channel_membership.channelId and channel.channelId = channel_extra.channelId" +
            " and channel_membership.userId = :userId" +
            " and channel.channelType in (:channelTypes)" +
            " and case when :isFilterByTags then channel.channelId in (SELECT channelId from channel_tag where tagName in (:includingTags))" +
            " else channel.channelId is not null end" + // always true
            " and case when :isFilterByMemberships then channel.channelId in (SELECT channelId from channel_membership where membership in (:memberships) and userId = :userId)" +
            " else channel.channelId is not null end" + // always true
            " and channel.channelId not in (SELECT channelId from channel_tag where tagName in (:excludingTags))" +
            " and case when :isDeleted is not null then channel.isDeleted = :isDeleted" +
            " else channel.channelId is not null end" + // always true
            " order by channel.lastActivity DESC")
    abstract DataSource.Factory<Integer, EkoChannelWithMembershipAndExtra> getDataSourceImpl(String userId, String[] channelTypes, boolean isFilterByTags, String[] includingTags, String[] excludingTags, boolean isFilterByMemberships, String[] memberships, Boolean isDeleted);

    public DataSource.Factory<Integer, EkoChannelEntity> getDataSource(String userId, String[] channelTypes, AmityTags includingTags, AmityTags excludingTags, List<String> memberships, Boolean isDeleted) {
        return getDataSourceImpl(userId,
                channelTypes,
                !includingTags.isEmpty(),
                includingTags.toArray(new String[0]),
                excludingTags.toArray(new String[0]),
                !memberships.isEmpty(),
                memberships.toArray(new String[0]),
                isDeleted).map(input -> input);
    }

    @Query("SELECT *" +
            " from channel" +
            " where channel.channelType in (:types)" +
            " and case when :isFilterByMemberships then channel.channelId in (SELECT channelId from channel_membership where membership in (:memberships) and userId = :userId)" +
            " else channel.channelId is not null end" + // always true
            " and case when :isIncludingTags then channel.channelId in (SELECT channelId from channel_tag where tagName in (:includingTags))" +
            " else channel.channelId is not null end" + // always true
            " and case when :isExcludingTags then channel.channelId not in (SELECT channelId from channel_tag where tagName in (:excludingTags))" +
            " else channel.channelId is not null end" + // always true
            " and case when :isDeleted is not null then channel.isDeleted = :isDeleted" +
            " else channel.channelId is not null end" + // always true
            " and channel.updatedAt > :now" +
            " and channel.channelId not in " +
            "(" +
            "SELECT amity_paging_id.id" +
            " from amity_paging_id" +
            " where amity_paging_id.hash = (:hash)" +
            " and amity_paging_id.nonce = (:nonce) " +
            ")" +
            " order by channel.updatedAt  desc" +
            " limit 1"
    )
    abstract Flowable<EkoChannelEntity> getLatestChannelImpl(String[] types,
                                                             boolean isFilterByMemberships,
                                                             String userId,
                                                             String[] memberships,
                                                             boolean isIncludingTags,
                                                             String[] includingTags,
                                                             boolean isExcludingTags,
                                                             String[] excludingTags,
                                                             Boolean isDeleted,
                                                             int hash,
                                                             int nonce,
                                                             DateTime now);

    public Flowable<EkoChannelEntity> getLatestChannel(String[] types,
                                                       String userId,
                                                       String[] memberships,
                                                       String[] includingTags,
                                                       String[] excludingTags,
                                                       Boolean isDeleted,
                                                       int hash,
                                                       int nonce,
                                                       DateTime now) {
        return getLatestChannelImpl(types,
                memberships.length > 0,
                userId,
                memberships,
                includingTags.length > 0,
                includingTags,
                excludingTags.length > 0,
                excludingTags,
                isDeleted,
                hash,
                nonce,
                now);
    }

    @Query("SELECT channel.*," +
            " channel_membership.readToSegment as membership_readToSegment," +
            " channel_extra.localReadToSegment as extra_localReadToSegment," +
            " channel_membership.lastMentionedSegment as membership_lastMentionedSegment," +
            " channel_extra.localLastMentionedSegment as extra_localLastMentionedSegment" +
            " from channel, channel_membership, channel_extra" +
            " where channel.channelId = channel_membership.channelId and channel.channelId = channel_extra.channelId" +
            " and channel.channelId = :channelId LIMIT 1")
    abstract Flowable<EkoChannelWithMembershipAndExtra> getChannelImpl(String channelId);

    public Flowable<EkoChannelEntity> getChannel(String channelId) {
        return getChannelImpl(channelId).map(input -> input);
    }

    @Query("SELECT channel.*," +
            " channel_membership.readToSegment as membership_readToSegment," +
            " channel_extra.localReadToSegment as extra_localReadToSegment," +
            " channel_membership.lastMentionedSegment as membership_lastMentionedSegment," +
            " channel_extra.localLastMentionedSegment as extra_localLastMentionedSegment" +
            " from channel, channel_membership, channel_extra" +
            " where channel.channelId = channel_membership.channelId and channel.channelId = channel_extra.channelId" +
            " and channel.channelId = :channelId LIMIT 1")
    abstract EkoChannelWithMembershipAndExtra getByIdNowImpl(String channelId);

    @Override
    public EkoChannelEntity getByIdNow(@NonNull String id) {
        return getByIdNowImpl(id);
    }

    @Query("SELECT channel.*," +
            " channel_membership.readToSegment as membership_readToSegment," +
            " channel_extra.localReadToSegment as extra_localReadToSegment," +
            " channel_membership.lastMentionedSegment as membership_lastMentionedSegment," +
            " channel_extra.localLastMentionedSegment as extra_localLastMentionedSegment" +
            " from channel, channel_membership, channel_extra" +
            " where channel.channelId = channel_membership.channelId and channel.channelId = channel_extra.channelId" +
            " and channel.channelId  IN (:ids)")
    abstract List<EkoChannelWithMembershipAndExtra> getByIdsNowImpl(List<String> ids);


    @Override
    @NonNull
    public List<EkoChannelEntity> getByIdsNow(@NonNull List<String> ids) {
        return FluentIterable.from(getByIdsNowImpl(ids))
                .transform(new Function<EkoChannelWithMembershipAndExtra, EkoChannelEntity>() {
                    @NullableDecl
                    @Override
                    public EkoChannelEntity apply(@NullableDecl EkoChannelWithMembershipAndExtra input) {
                        return input;
                    }
                }).toList();
    }

    @Query("UPDATE channel set lastActivity = :lastActivity where channelId = :channelId")
    public abstract void updateLastActivity(String channelId, DateTime lastActivity);

    // dummy update
    @Query("UPDATE channel set channelId = :channelId where channelId = :channelId")
    public abstract void updateUnreadCount(String channelId);

    @Query("UPDATE channel set messageCount = :messageCount where channelId = :channelId and messageCount < :messageCount")
    public abstract void updateMessageCount(String channelId, int messageCount);

    @Query("DELETE from channel")
    public abstract void deleteAll();

    @Query("DELETE from channel where channelId = :channelId")
    abstract void deleteByIdImpl(String channelId);

    @Transaction
    public void deleteById(String channelId) {
        deleteByIdImpl(channelId);
        messageDao.deleteAllFromChannel(channelId);
        channelMembershipDao.deleteAllFromChannel(channelId);
        channelExtraDao.deleteAllFromChannel(channelId);
    }

    @Query("SELECT channelId " +
            "from channel " +
            "where channelId in (SELECT channelId from channel_membership where membership = 'member' and userId = :userId)")
    public abstract Flowable<List<String>> getActiveIds(String userId);

    @Query("SELECT channelId" +
            " from channel" +
            " where channelId in (SELECT channelId from channel_membership where (membership = 'none' or membership = 'banned') and userId = :userId)")
    abstract List<String> getInactiveIds(String userId);

    @Transaction
    public void deleteAllLocallyInactiveChannelsAndUpdateAllActiveChannelsToNotReading() {
        List<String> inactiveChannelIds = getInactiveIds(AmityCoreClient.INSTANCE.getUserId());

        for (String inactiveChannelId : inactiveChannelIds) {
            deleteById(inactiveChannelId);
        }

        channelExtraDao.updateAllReadStatuses(EkoChannelReadStatus.NOT_READING);
    }

    private void beforeInsertOrUpdate(EkoChannelEntity channel) {
        EkoChannelEntity c = getByIdNow(channel.getChannelId());
        if (c != null) {
            channel.setMessageCount(Math.max(c.getMessageCount(), channel.getMessageCount()));
        }
    }

    private void beforeInsertOrUpdate(List<? extends EkoChannelEntity> cs) {
        for (EkoChannelEntity channel : cs) {
            beforeInsertOrUpdate(channel);
        }
    }

    @Transaction
    @Override
    public void insert(EkoChannelEntity channel) {
        beforeInsertOrUpdate(channel);
        super.insert(channel);
        EkoTagDao.update(channel, channelTagDao, EkoChannelTag::create);
        channelExtraDao.insertOrUpdate(channel.getChannelId());
    }

    @Transaction
    @Override
    public void insert(List<? extends EkoChannelEntity> channels) {
        beforeInsertOrUpdate(channels);
        super.insert(channels);
        EkoTagDao.update(channels, channelTagDao, EkoChannelTag::create);
        channelExtraDao.insertOrUpdate(FluentIterable.from(channels)
                .transform(channel -> channel.getChannelId()
                ).toList());
    }

    @Transaction
    @Override
    public void update(EkoChannelEntity channel) {
        beforeInsertOrUpdate(channel);
        super.update(channel);
        EkoTagDao.update(channel, channelTagDao, EkoChannelTag::create);
        channelExtraDao.insertOrUpdate(channel.getChannelId());
    }

}
