package com.ekoapp.ekosdk.internal.data.dao;

import androidx.annotation.NonNull;
import androidx.paging.DataSource;
import androidx.room.Dao;
import androidx.room.Query;
import androidx.room.Transaction;

import com.amity.socialcloud.sdk.social.community.AmityCommunitySortOption;
import com.ekoapp.ekosdk.internal.data.UserDatabase;
import com.ekoapp.ekosdk.internal.data.model.EkoCategoryEntity;
import com.ekoapp.ekosdk.internal.entity.CommunityEntity;

import java.util.List;

import io.reactivex.rxjava3.core.Completable;
import io.reactivex.rxjava3.core.Flowable;


@Dao
public abstract class EkoCommunityDao extends EkoObjectDao<CommunityEntity> {

    private final EkoCommunityWithCategoryDao communityWithCategoryDao = UserDatabase.get().communityWithCategoryDao();

    @Query("DELETE from community")
    public abstract void deleteAll();

    @Query("UPDATE community set isDeleted = 1 where communityId = :communityId")
    abstract Completable deleteByCommunityIdImpl(String communityId);

    public Completable deleteByCommunityId(String communityId) {
        return deleteByCommunityIdImpl(communityId);
    }

    @Query("SELECT *" +
            " from community" +
            " where community.communityId = :communityId" +
            " LIMIT 1")
    abstract Flowable<CommunityEntity> getByCommunityIdImpl(String communityId);

    public Flowable<CommunityEntity> getByCommunityId(String communityId) {
        return getByCommunityIdImpl(communityId);
    }

    @Query("SELECT *" +
            " from community" +
            " where community.communityId in (:communityIds)")
    abstract Flowable<List<CommunityEntity>> observeByCommunityIdsImpl(List<String> communityIds);

    public Flowable<List<CommunityEntity>> observeByCommunityIds(List<String> communityIds) {
        return observeByCommunityIdsImpl(communityIds);
    }

    @Query("SELECT *" +
            " from community" +
            " where community.displayName LIKE :keyword || '%' COLLATE NOCASE" +
            " and community.communityId in (SELECT communityId from category where categoryId = :categoryId)" +
            " and community.isDeleted = " +
            "(case when :isDeleted is null then isDeleted else :isDeleted end)" +
            " and community.isJoined = " +
            "(case when community.isPublic is 1 then isJoined else 1 end)" +
            " order by case when :isSortByDisplayName = 1 then community.displayName end asc," +
            " case when :isSortByDisplayName = 0 and :isSortByCreatedACS = 1 then community.createdAt end asc," +
            " case when :isSortByDisplayName = 0 and :isSortByCreatedACS = 0 then community.createdAt end desc")
    abstract DataSource.Factory<Integer, CommunityEntity> getAllByCategoryIdImpl(
            String keyword,
            String categoryId,
            boolean isSortByDisplayName,
            boolean isSortByCreatedACS,
            Boolean isDeleted);

    public DataSource.Factory<Integer, CommunityEntity> getAllByCategoryId(String keyword,
                                                                           String categoryId,
                                                                           AmityCommunitySortOption sortBy,
                                                                           Boolean isDeleted) {
        return getAllByCategoryIdImpl(
                keyword,
                categoryId,
                isSortByDisplayName(sortBy),
                isSortByCreatedACS(sortBy),
                isDeleted)
                .map(input -> input);
    }

    @Query("SELECT *" +
            " from community" +
            " where community.displayName LIKE :keyword || '%' COLLATE NOCASE" +
            " and community.isJoined = 1" +
            " and community.communityId in (SELECT communityId from category where categoryId = :categoryId)" +
            " and community.isDeleted = " +
            "(case when :isDeleted is null then isDeleted else :isDeleted end)" +
            " order by case when :isSortByDisplayName = 1 then community.displayName end asc," +
            " case when :isSortByDisplayName = 0 and :isSortByCreatedACS = 1 then community.createdAt end asc," +
            " case when :isSortByDisplayName = 0 and :isSortByCreatedACS = 0 then community.createdAt end desc")
    abstract DataSource.Factory<Integer, CommunityEntity> getAllByCategoryIdForMemberImpl(
            String keyword,
            String categoryId,
            boolean isSortByDisplayName,
            boolean isSortByCreatedACS,
            Boolean isDeleted);

    public DataSource.Factory<Integer, CommunityEntity> getAllByCategoryIdForMember(String keyword,
                                                                                    String categoryId,
                                                                                    AmityCommunitySortOption sortBy,
                                                                                    Boolean isDeleted) {
        return getAllByCategoryIdForMemberImpl(
                keyword,
                categoryId,
                isSortByDisplayName(sortBy),
                isSortByCreatedACS(sortBy),
                isDeleted)
                .map(input -> input);
    }

    @Query("SELECT *" +
            " from community" +
            " where community.displayName LIKE :keyword || '%' COLLATE NOCASE" +
            " and community.isJoined = 0" +
            " and community.communityId in (SELECT communityId from category where categoryId = :categoryId)" +
            " and community.isDeleted = " +
            "(case when :isDeleted is null then isDeleted else :isDeleted end)" +
            " order by case when :isSortByDisplayName = 1 then community.displayName end asc," +
            " case when :isSortByDisplayName = 0 and :isSortByCreatedACS = 1 then community.createdAt end asc," +
            " case when :isSortByDisplayName = 0 and :isSortByCreatedACS = 0 then community.createdAt end desc")
    abstract DataSource.Factory<Integer, CommunityEntity> getAllByCategoryIdForNonMemberImpl(
            String keyword,
            String categoryId,
            boolean isSortByDisplayName,
            boolean isSortByCreatedACS,
            Boolean isDeleted);

    public DataSource.Factory<Integer, CommunityEntity> getAllByCategoryIdForNonMember(String keyword,
                                                                                       String categoryId,
                                                                                       AmityCommunitySortOption sortBy,
                                                                                       Boolean isDeleted) {
        return getAllByCategoryIdForNonMemberImpl(
                keyword,
                categoryId,
                isSortByDisplayName(sortBy),
                isSortByCreatedACS(sortBy),
                isDeleted)
                .map(input -> input);
    }

    @Query("SELECT *" +
            " from community" +
            " where community.displayName LIKE :keyword || '%' COLLATE NOCASE" +
            " and community.isDeleted = " +
            "(case when :isDeleted is null then isDeleted else :isDeleted end)" +
            " and community.isJoined = " +
            "(case when community.isPublic is 1 then isJoined else 1 end)" +
            " order by case when :isSortByDisplayName = 1 then community.displayName end asc," +
            " case when :isSortByDisplayName = 0 and :isSortByCreatedACS = 1 then community.createdAt end asc," +
            " case when :isSortByDisplayName = 0 and :isSortByCreatedACS = 0 then community.createdAt end desc")
    abstract DataSource.Factory<Integer, CommunityEntity> getAllByKeywordImpl(
            String keyword,
            boolean isSortByDisplayName,
            boolean isSortByCreatedACS,
            Boolean isDeleted);

    public DataSource.Factory<Integer, CommunityEntity> getAllByKeyword(String keyword,
                                                                        AmityCommunitySortOption sortBy,
                                                                        Boolean isDeleted) {
        return getAllByKeywordImpl(
                keyword,
                isSortByDisplayName(sortBy),
                isSortByCreatedACS(sortBy),
                isDeleted)
                .map(input -> input);
    }

    @Query("SELECT *" +
            " from community" +
            " where community.displayName LIKE :keyword || '%' COLLATE NOCASE" +
            " and community.isJoined = 1" +
            " and community.isDeleted = " +
            "(case when :isDeleted is null then isDeleted else :isDeleted end)" +
            " order by case when :isSortByDisplayName = 1 then community.displayName end asc," +
            " case when :isSortByDisplayName = 0 and :isSortByCreatedACS = 1 then community.createdAt end asc," +
            " case when :isSortByDisplayName = 0 and :isSortByCreatedACS = 0 then community.createdAt end desc")
    abstract DataSource.Factory<Integer, CommunityEntity> getAllByKeywordForMemberImpl(
            String keyword,
            boolean isSortByDisplayName,
            boolean isSortByCreatedACS,
            Boolean isDeleted);

    public DataSource.Factory<Integer, CommunityEntity> getAllByKeywordForMember(String keyword,
                                                                                 AmityCommunitySortOption sortBy,
                                                                                 Boolean isDeleted) {
        return getAllByKeywordForMemberImpl(
                keyword,
                isSortByDisplayName(sortBy),
                isSortByCreatedACS(sortBy),
                isDeleted)
                .map(input -> input);
    }

    @Query("SELECT *" +
            " from community" +
            " where community.displayName LIKE :keyword || '%' COLLATE NOCASE" +
            " and community.isJoined = 0" +
            " and community.isDeleted = " +
            "(case when :isDeleted is null then isDeleted else :isDeleted end)" +
            " order by case when :isSortByDisplayName = 1 then community.displayName end asc," +
            " case when :isSortByDisplayName = 0 and :isSortByCreatedACS = 1 then community.createdAt end asc," +
            " case when :isSortByDisplayName = 0 and :isSortByCreatedACS = 0 then community.createdAt end desc")
    abstract DataSource.Factory<Integer, CommunityEntity> getAllByKeywordForNonMemberImpl(
            String keyword,
            boolean isSortByDisplayName,
            boolean isSortByCreatedACS,
            Boolean isDeleted);

    public DataSource.Factory<Integer, CommunityEntity> getAllByKeywordForNonMember(String keyword,
                                                                                    AmityCommunitySortOption sortBy,
                                                                                    Boolean isDeleted) {
        return getAllByKeywordForNonMemberImpl(
                keyword,
                isSortByDisplayName(sortBy),
                isSortByCreatedACS(sortBy),
                isDeleted)
                .map(input -> input);
    }

    @Query("UPDATE community set isJoined = :isJoined where communityId = :communityId")
    abstract void updateIsJoinedImpl(String communityId, Boolean isJoined);

    public void updateIsJoined(String communityId, Boolean isJoined) {
        updateIsJoinedImpl(communityId, isJoined);
    }

    private Boolean isSortByDisplayName(AmityCommunitySortOption sortBy) {
        return sortBy == AmityCommunitySortOption.DISPLAY_NAME;
    }

    private Boolean isSortByCreatedACS(AmityCommunitySortOption sortBy) {
        return sortBy == AmityCommunitySortOption.FIRST_CREATED;
    }

    @Transaction
    @Override
    public void insert(CommunityEntity community) {
        super.insert(community);
        EkoCategoryDao.update(community, communityWithCategoryDao, EkoCategoryEntity::create);
    }


    @Transaction
    @Override
    public void insert(@NonNull List<? extends CommunityEntity> communities) {
        super.insert(communities);
        EkoCategoryDao.update(communities, communityWithCategoryDao, EkoCategoryEntity::create);
    }

    @Transaction
    @Override
    public void update(CommunityEntity community) {
        super.update(community);
        EkoCategoryDao.update(community, communityWithCategoryDao, EkoCategoryEntity::create);
    }

    @Transaction
    @Override
    public void update(@NonNull List<? extends CommunityEntity> communities) {
        super.update(communities);
        EkoCategoryDao.update(communities, communityWithCategoryDao, EkoCategoryEntity::create);
    }

    @Query("SELECT * from community where communityId = :communityId LIMIT 1")
    abstract CommunityEntity getByIdNowImpl(String communityId);

    public CommunityEntity getByIdNow(String communityId) {
        return getByIdNowImpl(communityId);
    }

    @Query("SELECT *" +
            " from community" +
            " where community.communityId IN (:ids)")
    abstract List<CommunityEntity> getByIdsNowImpl(List<String> ids);

    public List<CommunityEntity> getByIdNow(List<String> ids) {
        return getByIdsNowImpl(ids);
    }

    @Transaction
    @Query("SELECT community.*, trending_community.*" +
            " from community, trending_community" +
            " where community.communityId = trending_community.communityId" +
            " order by trending_community.createdAt ASC")
    abstract DataSource.Factory<Integer, CommunityEntity> getTrendingCommunityCollectionImpl();

    public DataSource.Factory<Integer, CommunityEntity> getTrendingCommunityCollection() {
        return getTrendingCommunityCollectionImpl();
    }

    @Transaction
    @Query("SELECT community.*, recommended_community.*" +
            " from community, recommended_community" +
            " where community.communityId = recommended_community.communityId" +
            " order by recommended_community.createdAt ASC")
    abstract DataSource.Factory<Integer, CommunityEntity> getRecommendedCommunityCollectionImpl();

    public DataSource.Factory<Integer, CommunityEntity> getRecommendedCommunityCollection() {
        return getRecommendedCommunityCollectionImpl();
    }

    @Transaction
    public void updateIsJoined(String communityId) {
        updateIsJoinedImpl(communityId);
    }

    @Query("UPDATE community set communityId = :communityId where communityId = :communityId")
    abstract void updateIsJoinedImpl(String communityId);

}