package com.ekoapp.ekosdk.internal.data.dao;

import androidx.paging.DataSource;
import androidx.room.Dao;
import androidx.room.Delete;
import androidx.room.Query;
import androidx.room.Transaction;

import com.amity.socialcloud.sdk.AmityCoreClient;
import com.amity.socialcloud.sdk.social.community.AmityCommunityMembershipSortOption;
import com.ekoapp.ekosdk.internal.data.UserDatabase;
import com.ekoapp.ekosdk.internal.data.model.CommunityMembershipPermissionEntity;
import com.ekoapp.ekosdk.internal.data.model.CommunityMembershipRoleEntity;
import com.ekoapp.ekosdk.internal.entity.CommunityMembershipEntity;
import com.google.common.base.Objects;

import org.jetbrains.annotations.NotNull;

import java.util.List;

import io.reactivex.Flowable;

@Dao
public abstract class EkoCommunityMembershipDao extends EkoObjectDao<CommunityMembershipEntity> {

    private final EkoCommunityDao communityDao;
    private final EkoCommunityRoleDao communityRoleDao;
    private final EkoCommunityPermissionDao communityPermissionDao;

    public EkoCommunityMembershipDao() {
        UserDatabase db = UserDatabase.get();
        communityRoleDao = db.communityRoleDao();
        communityPermissionDao = db.communityPermissionDao();
        communityDao = db.communityDao();
    }

    @Delete
    public abstract void delete(CommunityMembershipEntity communityMembership);

    @Query("DELETE from community_membership")
    public abstract void deleteAll();

    @Query("DELETE from community_membership where community_membership.communityId = :communityId")
    abstract void deleteByCommunityIdImpl(String communityId);

    @Query("SELECT * from community_membership where communityId = :communityId and userId = :userId LIMIT 1")
    abstract CommunityMembershipEntity getByCommunityIdAndUserIdNowImpl(String communityId, String userId);

    public CommunityMembershipEntity getByCommunityIdAndUserIdNow(String communityId, String userId) {
        return getByCommunityIdAndUserIdNowImpl(communityId, userId);
    }

    @Query("SELECT * from community_membership where id = :id LIMIT 1")
    abstract CommunityMembershipEntity getByIdNowImpl(String id);

    @Override
    public CommunityMembershipEntity getByIdNow(String id) {
        return getByIdNowImpl(id);
    }

    @Query("SELECT *" +
            " from community_membership" +
            " where communityId = :communityId" +
            " and userId = :userId" +
            " LIMIT 1")
    abstract Flowable<CommunityMembershipEntity> getByIdImpl(String communityId, String userId);

    public Flowable<CommunityMembershipEntity> getById(String communityId, String userId) {
        return getByIdImpl(communityId, userId);
    }

    @Query("SELECT * from community_membership where userId in (:ids)")
    abstract List<CommunityMembershipEntity> getByIdsNowImpl(@NotNull List<String> ids);

    @Override
    public List<CommunityMembershipEntity> getByIdsNow(@NotNull List<String> ids) {
        return getByIdsNowImpl(ids);
    }

    @Transaction
    public void deleteByCommunityId(String communityId) {
        deleteByCommunityIdImpl(communityId);
    }

    @Query("SELECT *" +
            " from community_membership, community_member_query_token" +
            " where community_membership.communityId = :communityId" +
            " and community_membership.communityMembership in (:memberships)" +
            " and userId in (SELECT userId from community_role where communityId = :communityId and roleName in (:role))" +
            " and" +
            " (community_member_query_token.communityId = :communityId" +
            " and community_member_query_token.roles = :queryRoles" +
            " and community_member_query_token.memberships = :queryMemberships" +
            " and community_member_query_token.sortOption = :querySortOption" +
            " and community_member_query_token.pageNumber = 1" +
            " and community_member_query_token.ids like '%' || community_membership.id || '%')" +
            " order by case when :isSortByCreatedACS = 1 then community_membership.createdAt end asc," +
            " case when :isSortByCreatedACS = 0 then community_membership.createdAt end desc")
    abstract DataSource.Factory<Integer, CommunityMembershipEntity> getByMembershipsAndRolesImpl(String communityId,
                                                                                                 List<String> role,
                                                                                                 List<String> memberships,
                                                                                                 boolean isSortByCreatedACS,
                                                                                                 String queryRoles,
                                                                                                 String queryMemberships,
                                                                                                 String querySortOption);

    @Query("SELECT *" +
            " from community_membership, community_member_query_token" +
            " where community_membership.communityId = :communityId" +
            " and community_membership.communityMembership in (:memberships)" +
            " and" +
            " (community_member_query_token.communityId = :communityId" +
            " and community_member_query_token.roles = :queryRoles" +
            " and community_member_query_token.memberships = :queryMemberships" +
            " and community_member_query_token.sortOption = :querySortOption" +
            " and community_member_query_token.pageNumber = 1" +
            " and community_member_query_token.ids like '%' || community_membership.id || '%')" +
            " order by case when :isSortByCreatedACS = 1 then community_membership.createdAt end asc," +
            " case when :isSortByCreatedACS = 0 then community_membership.createdAt end desc")
    abstract DataSource.Factory<Integer, CommunityMembershipEntity> getByMembershipsImpl(String communityId,
                                                                                         List<String> memberships,
                                                                                         boolean isSortByCreatedACS,
                                                                                         String queryRoles,
                                                                                         String queryMemberships,
                                                                                         String querySortOption);

    public DataSource.Factory<Integer, CommunityMembershipEntity> getMembers(String communityId,
                                                                             List<String> roles,
                                                                             List<String> memberships,
                                                                             AmityCommunityMembershipSortOption sortBy) {
        if (roles == null || roles.isEmpty()) {
            return getByMembershipsImpl(communityId, memberships, isSortByCreatedACS(sortBy), "", memberships.toString(), sortBy.getApiKey());
        } else {
            return getByMembershipsAndRolesImpl(communityId, roles, memberships, isSortByCreatedACS(sortBy), roles.toString(), memberships.toString(), sortBy.getApiKey());
        }
    }

    private Boolean isSortByCreatedACS(AmityCommunityMembershipSortOption sortBy) {
        return sortBy == AmityCommunityMembershipSortOption.FIRST_CREATED;
    }

    @Transaction //dummy update, for triggering user update tied to community member.
    public void updateUser(String userId) {
        updateUserImpl(userId);
    }

    @Query("UPDATE channel_membership set userId = :userId where userId = :userId")
    abstract void updateUserImpl(String userId);


    @Transaction
    @Override
    public void insert(CommunityMembershipEntity membership) {
        super.insert(membership);
        EkoRoleDao.update(membership, communityRoleDao, CommunityMembershipRoleEntity::create);
        EkoPermissionDao.update(membership, communityPermissionDao, CommunityMembershipPermissionEntity::create);
        updateIsJoined(membership);
    }

    @Transaction
    @Override
    public void insert(List<? extends CommunityMembershipEntity> memberships) {
        super.insert(memberships);
        EkoRoleDao.update(memberships, communityRoleDao, CommunityMembershipRoleEntity::create);
        EkoPermissionDao.update(memberships, communityPermissionDao, CommunityMembershipPermissionEntity::create);
        for (CommunityMembershipEntity membership : memberships) {
            updateIsJoined(membership);
        }
    }

    @Override
    public void update(CommunityMembershipEntity membership) {
        super.update(membership);
        EkoRoleDao.update(membership, communityRoleDao, CommunityMembershipRoleEntity::create);
        EkoPermissionDao.update(membership, communityPermissionDao, CommunityMembershipPermissionEntity::create);
        updateIsJoined(membership);
    }

    private void updateIsJoined(CommunityMembershipEntity membership) {
        if (Objects.equal(membership.getUserId(), AmityCoreClient.INSTANCE.getUserId())) {
            communityDao.updateIsJoined(membership.getCommunityId());
        }
    }

}
