package com.ekoapp.ekosdk.internal.data.dao

import androidx.paging.PagingSource
import androidx.room.Dao
import androidx.room.Insert
import androidx.room.OnConflictStrategy
import androidx.room.RawQuery
import androidx.sqlite.db.SimpleSQLiteQuery
import co.amity.rxremotemediator.AmityPagingDao
import co.amity.rxremotemediator.AmityPagingId
import com.amity.socialcloud.sdk.core.permission.AmityRoles
import com.ekoapp.ekosdk.community.membership.query.AmityCommunityMembership
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.entity.CommunityMembershipEntity
import com.ekoapp.ekosdk.internal.keycreator.CommunityMembershipKeyCreator
import com.ekoapp.ekosdk.internal.paging.QueryStreamPagingDao
import io.reactivex.Completable

@Dao
internal interface EkoCommunityMembershipPagingDao : QueryStreamPagingDao<CommunityMembershipEntity> {

    @RawQuery(observedEntities = [CommunityMembershipEntity::class, AmityPagingId::class])
    override fun queryPagingData(sqlQuery: SimpleSQLiteQuery): PagingSource<Int, CommunityMembershipEntity>

    fun searchCommunityMembership(
        communityId: String,
        roles: AmityRoles,
        filters: List<AmityCommunityMembership>?,
        keyword: String?
    ): PagingSource<Int, CommunityMembershipEntity> {
        return queryPagingData(
            generateSqlQuery(
                tableName = "community_membership",
                primaryKeyColumnName = "userId",
                additionalPrimaryKeys = mapOf("communityId" to communityId),
                queryParameters = mapOf(
                    "communityId" to communityId,
                    "roles" to roles,
                    "memberships" to (filters ?: listOf()),
                    "search" to (keyword ?: "")
                ),
                nonce = AmityNonce.COMMUNITY_MEMBERSHIP,
                order = AmityPagingDao.Order.ASC
            )
        )
    }
    
    fun queryCommunityMembership(
            communityId: String,
            roles: AmityRoles,
            filters: List<AmityCommunityMembership>?,
            sortBy: String?,
            keyword: String?
    ): PagingSource<Int, CommunityMembershipEntity> {
        return queryPagingData(
                generateQueryStreamSQL(
                        tableName = "community_membership",
                        primaryKeyColumnName = "userId",
                        additionalPrimaryKeys = mapOf("communityId" to communityId),
                        queryStreamKeyCreator = CommunityMembershipKeyCreator(communityId, roles, filters, sortBy, keyword),
                        nonce = AmityNonce.COMMUNITY_MEMBERSHIP,
                        order = AmityPagingDao.Order.ASC
                )
        )
    }

    @Insert(onConflict = OnConflictStrategy.REPLACE)
    fun insertCommunityMembership(entities: List<CommunityMembershipEntity>): Completable
}