package com.ekoapp.ekosdk.internal.data.dao;

import androidx.paging.DataSource;
import androidx.room.Dao;
import androidx.room.Query;

import com.amity.socialcloud.sdk.core.user.AmityFollowStatusFilter;
import com.ekoapp.ekosdk.internal.data.model.EkoUserFollowStatusEntity;

import org.joda.time.DateTime;

import java.util.List;

import io.reactivex.rxjava3.core.Flowable;

@Dao
public abstract class EkoFollowDao extends EkoObjectDao<EkoUserFollowStatusEntity> implements AmityPagingDao<EkoUserFollowStatusEntity> {

    @Query("UPDATE user_follow set status = :status where targetUserId = :userId")
    public abstract void updateFollowing(String userId, String status);

    @Query("UPDATE user_follow set status = :status where sourceUserId = :userId")
    public abstract void updateFollower(String userId, String status);

    @Query("DELETE from user_follow")
    public abstract void deleteAll();

    @Query("SELECT * from user_follow where id = :id LIMIT 1")
    abstract EkoUserFollowStatusEntity getByIdNowImpl(String id);

    public EkoUserFollowStatusEntity getByIdNow(String id) {
        return getByIdNowImpl(id);
    }

    @Query("SELECT * from user_follow where id IN (:ids)")
    abstract List<EkoUserFollowStatusEntity> getByIdsNowImpl(List<String> ids);

    public List<EkoUserFollowStatusEntity> getByIdsNow(List<String> ids) {
        return getByIdsNowImpl(ids);
    }

    @Query("SELECT * from user_follow" +
            " where sourceUserId = :sourceUserId" +
            " and targetUserId = :targetUserId" +
            " LIMIT 1")
    abstract Flowable<EkoUserFollowStatusEntity> getAllByIdImpl(String sourceUserId, String targetUserId);

    public Flowable<EkoUserFollowStatusEntity> getUserFollowById(String sourceUserId, String targetUserId) {
        return getAllByIdImpl(sourceUserId, targetUserId);
    }

    @Query("SELECT * from user_follow, follower_query_token" +
            " where user_follow.targetUserId = :userId" +
            " and user_follow.status in (:statuses)" +
            " and follower_query_token.userId = :userId" +
            " and follower_query_token.filter = :statusFilter" +
            " and follower_query_token.pageNumber = 1" +
            " and" +
            " ((follower_query_token.ids like '%' || user_follow.id || '%')" +
            " or" +
            " (user_follow.updatedAt > follower_query_token.updatedAt))")
    abstract DataSource.Factory<Integer, EkoUserFollowStatusEntity> getFollowersDataSourceImpl(String userId, List<String> statuses, String statusFilter);

    public DataSource.Factory<Integer, EkoUserFollowStatusEntity> getFollowersDataSource(String userId, AmityFollowStatusFilter statusFilter) {
        return getFollowersDataSourceImpl(userId, statusFilter.getStatuses(), statusFilter.getApiKey());
    }

    @Query("SELECT * from user_follow, following_query_token" +
            " where user_follow.sourceUserId = :userId" +
            " and user_follow.status in (:statuses)" +
            " and following_query_token.userId = :userId" +
            " and following_query_token.filter = :statusFilter" +
            " and following_query_token.pageNumber = 1" +
            " and" +
            " ((following_query_token.ids like '%' || user_follow.id || '%')" +
            " or" +
            " (user_follow.updatedAt > following_query_token.updatedAt))")
    abstract DataSource.Factory<Integer, EkoUserFollowStatusEntity> getFollowingsDataSourceImpl(String userId, List<String> statuses, String statusFilter);

    public DataSource.Factory<Integer, EkoUserFollowStatusEntity> getFollowingsDataSource(String userId, AmityFollowStatusFilter statusFilter) {
        return getFollowingsDataSourceImpl(userId, statusFilter.getStatuses(), statusFilter.getApiKey());
    }

    public Flowable<EkoUserFollowStatusEntity> getLatestFollowing(String userId, List<String> statuses) {
        return getLatestFollowingImpl(userId, statuses, DateTime.now());
    }

    @Query("SELECT * from user_follow" +
            " where user_follow.sourceUserId = :userId" +
            " and user_follow.status in (:statuses)" +
            " and user_follow.updatedAt > :now" +
            " order by user_follow.updatedAt  desc" +
            " limit 1"
    )
    abstract Flowable<EkoUserFollowStatusEntity> getLatestFollowingImpl(String userId, List<String> statuses, DateTime now);


    public Flowable<EkoUserFollowStatusEntity> getLatestFollower(String userId, List<String> statuses) {
        return getLatestFollowerImpl(userId, statuses, DateTime.now());
    }

    @Query("SELECT * from user_follow" +
            " where user_follow.targetUserId = :userId" +
            " and user_follow.status in (:statuses)" +
            " and user_follow.updatedAt > :now" +
            " order by user_follow.updatedAt  desc" +
            " limit 1"
    )
    abstract Flowable<EkoUserFollowStatusEntity> getLatestFollowerImpl(String userId, List<String> statuses, DateTime now);

}