package com.ekoapp.ekosdk.internal.data.dao;

import androidx.annotation.NonNull;
import androidx.paging.DataSource;
import androidx.room.Dao;
import androidx.room.Query;
import androidx.room.Transaction;

import com.amity.socialcloud.sdk.core.AmityTags;
import com.ekoapp.ekosdk.internal.EkoMessageEntity;
import com.ekoapp.ekosdk.internal.data.UserDatabase;
import com.ekoapp.ekosdk.internal.data.model.EkoMessageTag;

import org.joda.time.DateTime;

import java.util.List;

import io.reactivex.rxjava3.core.Completable;
import io.reactivex.rxjava3.core.Flowable;


@Dao
public abstract class EkoMessageDao extends EkoObjectDao<EkoMessageEntity> implements AmityPagingDao<EkoMessageEntity> {

    private final EkoMessageTagDao messageTagDao;

    EkoMessageDao() {
        this.messageTagDao = UserDatabase.get().messageTagDao();
    }

    @Deprecated
    @Query("SELECT count(*) from message")
    public abstract Flowable<Integer> getCount();

    @Deprecated
    @Query("SELECT count(*) from message where channelId = :channelId")
    public abstract Flowable<Integer> getCount(String channelId);

    @Deprecated
    @Query("SELECT message.*," +
            " message_flag.messageId as flag_messageId," +
            " message_flag.flag as flag_flag," +
            " message_flag.localFlag as flag_localFlag" +
            " from message, message_flag" +
            " where message.channelId = :channelId" +
            " and message.messageId = message_flag.messageId" +
            " and case when :isFilterByParentId and :parentId is not null then message.parentId = :parentId " +
            " when :isFilterByParentId and :parentId is null then message.parentId is null " +
            " else message.messageId is not null end" + // always true
            " and case when :isFilterByTags then message.messageId in (SELECT messageId from message_tag where tagName in (:includingTags)) " +
            " else message.messageId is not null end" + // always true
            " and message.messageId not in (SELECT messageId from message_tag where tagName in (:excludingTags))" +
            " and case when :isDeleted is not null then message.isDeleted = :isDeleted" +
            " else message.messageId is not null end" + // always true
            " order by channelSegment, createdAt ASC")
    abstract DataSource.Factory<Integer, EkoMessageEntity> getDataSourceImpl(String channelId, boolean isFilterByParentId, String parentId, boolean isFilterByTags, String[] includingTags, String[] excludingTags, Boolean isDeleted);

    public DataSource.Factory<Integer, EkoMessageEntity> getDataSource(String channelId, boolean isFilterByParentId, String parentId, AmityTags includingTags, AmityTags excludingTags, Boolean isDeleted) {
        return getDataSourceImpl(channelId,
                isFilterByParentId,
                parentId,
                !includingTags.isEmpty(),
                includingTags.toArray(new String[0]),
                excludingTags.toArray(new String[0]),
                isDeleted);
    }

    @Query("SELECT message.*," +
            " message_flag.messageId as flag_messageId," +
            " message_flag.flag as flag_flag," +
            " message_flag.localFlag as flag_localFlag" +
            " from message, message_flag" +
            " where message.channelId = :channelId" +
            " and message.messageId = message_flag.messageId" +
            " and message.isDeleted = case when :isDeleted is not null then :isDeleted else isDeleted end" +
            " order by channelSegment DESC LIMIT 1")
    abstract Flowable<EkoMessageEntity> getLatestMessageImpl(String channelId, Boolean isDeleted);

    public Flowable<EkoMessageEntity> getLatestMessage(String channelId) {
        return getLatestMessageImpl(channelId, null);
    }

    public Flowable<EkoMessageEntity> getLatestMessage(String channelId, Boolean isDeleted) {
        return getLatestMessageImpl(channelId, isDeleted);
    }

    @Query("SELECT channelSegment from message where channelId = :channelId and syncState = 'synced' order by channelSegment DESC LIMIT 1")
    public abstract int getHighestChannelSegment(String channelId);

    @Query("UPDATE message set syncState = 'failed' where syncState = 'syncing'")
    public abstract void cleanUpSyncingStateOnStartup();

    @Query("UPDATE message set syncState = 'failed' where syncState = 'uploading'")
    public abstract void cleanUpUploadingStateOnStartup();

    @Query("UPDATE message set syncState = :state where messageId = :messageId")
    public abstract Completable updateSyncState(String messageId, String state);

    @Query("DELETE from message")
    public abstract void deleteAll();

    @Query("DELETE from message where channelId = :channelId")
    public abstract void deleteAllFromChannel(String channelId);

    @Query("SELECT * from message where channelId = :channelId order by createdAt DESC LIMIT -1 OFFSET :offset")
    abstract List<EkoMessageEntity> getOldMessages(String channelId, int offset);

    @Transaction
    public void retainLatestFromChannel(String channelId, int offset) {
        List<EkoMessageEntity> oldMessages = getOldMessages(channelId, offset);
        delete(oldMessages);
    }

    @Query("DELETE from message where messageId = :messageId")
    public abstract Completable hardDeleteMessage(String messageId);

    @Query("DELETE from message where channelId = :channelId")
    public abstract void hardDeleteAllFromChannel(String channelId);

    @Query("UPDATE message set isDeleted = 1, data = null where channelId = :channelId and userId = :userId")
    abstract void softDeleteFromChannelByUserIdImpl(String channelId, String userId);

    @Transaction
    public void softDeleteFromChannelByUserId(String channelId, String userId) {
        softDeleteFromChannelByUserIdImpl(channelId, userId);
    }

    @Query("UPDATE message set userId = :userId where userId = :userId")
    abstract void updateUserImpl(String userId);

    @Transaction //dummy update, for triggering all messages sent by this userId.
    public void updateUser(String userId) {
        updateUserImpl(userId);
    }

    @Transaction
    @Override
    public void insert(EkoMessageEntity message) {
        super.insert(message);
        EkoTagDao.update(message, messageTagDao, EkoMessageTag::create);
    }

    @Transaction
    @Override
    public void insert(List<? extends EkoMessageEntity> messages) {
        super.insert(messages);
        EkoTagDao.update(messages, messageTagDao, EkoMessageTag::create);
    }

    @Transaction
    @Override
    public void update(EkoMessageEntity message) {
        super.update(message);
        EkoTagDao.update(message, messageTagDao, EkoMessageTag::create);
    }

    @Query("SELECT message.*," +
            " message_flag.messageId as flag_messageId," +
            " message_flag.flag as flag_flag," +
            " message_flag.localFlag as flag_localFlag" +
            " from message, message_flag" +
            " where message.messageId = :messageId" +
            " and message.messageId = message_flag.messageId" +
            " LIMIT 1")
    abstract Flowable<EkoMessageEntity> getByIdImpl(String messageId);

    public Flowable<EkoMessageEntity> getById(String messageId) {
        return getByIdImpl(messageId);
    }

    @Query("SELECT message.*," +
            " message_flag.messageId as flag_messageId," +
            " message_flag.flag as flag_flag," +
            " message_flag.localFlag as flag_localFlag" +
            " from message, message_flag" +
            " where message.messageId = :messageId" +
            " and message.messageId = message_flag.messageId" +
            " LIMIT 1")
    abstract EkoMessageEntity getByIdNowImpl(String messageId);

    @Override
    public EkoMessageEntity getByIdNow(@NonNull String messageId) {
        return getByIdNowImpl(messageId);
    }

    @Query("SELECT message.*," +
            " message_flag.messageId as flag_messageId," +
            " message_flag.flag as flag_flag," +
            " message_flag.localFlag as flag_localFlag" +
            " from message, message_flag" +
            " where message.messageId IN (:messageIds)" +
            " and message.messageId = message_flag.messageId")
    abstract List<EkoMessageEntity> getByIdsNowImpl(List<String> messageIds);

    @NonNull
    @Override
    public List<EkoMessageEntity> getByIdsNow(@NonNull List<String> ids) {
        return getByIdsNowImpl(ids);
    }

    @Query("SELECT message.*," +
            " message_flag.messageId as flag_messageId," +
            " message_flag.flag as flag_flag," +
            " message_flag.localFlag as flag_localFlag" +
            " from message, message_flag" +
            " where message.channelId = :channelId" +
            " and message.messageId = message_flag.messageId" +
            " and case when :isFilterByParentId and :parentId is not null then message.parentId = :parentId " +
            " when :isFilterByParentId and :parentId is null then message.parentId is null " +
            " else message.messageId is not null end" + // always true
            " and case when :isFilterByTags then message.messageId in (SELECT messageId from message_tag where tagName in (:includingTags)) " +
            " else message.messageId is not null end" + // always true
            " and message.messageId not in (SELECT messageId from message_tag where tagName in (:excludingTags))" +
            " and case when :isDeleted is not null then message.isDeleted = :isDeleted" +
            " else message.messageId is not null end" + // always true
            " and message.type IN (:type)" +
            " order by channelSegment, createdAt ASC")
    abstract Flowable<List<EkoMessageEntity>> observeMessagesWithTypeImpl(String channelId, boolean isFilterByParentId, String parentId,
                                                                  boolean isFilterByTags, String[] includingTags,
                                                                  String[] excludingTags, Boolean isDeleted, List<String> type);

    @Query("SELECT message.*," +
            " message_flag.messageId as flag_messageId," +
            " message_flag.flag as flag_flag," +
            " message_flag.localFlag as flag_localFlag" +
            " from message, message_flag" +
            " where message.channelId = :channelId" +
            " and message.messageId = message_flag.messageId" +
            " and case when :isFilterByParentId and :parentId is not null then message.parentId = :parentId " +
            " when :isFilterByParentId and :parentId is null then message.parentId is null " +
            " else message.messageId is not null end" + // always true
            " and case when :isFilterByTags then message.messageId in (SELECT messageId from message_tag where tagName in (:includingTags)) " +
            " else message.messageId is not null end" + // always true
            " and message.messageId not in (SELECT messageId from message_tag where tagName in (:excludingTags))" +
            " and case when :isDeleted is not null then message.isDeleted = :isDeleted" +
            " else message.messageId is not null end" + // always true
            " order by channelSegment, createdAt ASC")
    abstract Flowable<List<EkoMessageEntity>> observeMessagesImpl(String channelId, boolean isFilterByParentId, String parentId,
                                                                      boolean isFilterByTags, String[] includingTags,
                                                                      String[] excludingTags, Boolean isDeleted);

    public Flowable<List<EkoMessageEntity>> observeMessages(String channelId, boolean isFilterByParentId,
                                                                String parentId, AmityTags includingTags, AmityTags excludingTags,
                                                                Boolean isDeleted, List<String> type) {
        if (type.isEmpty()) {
            return observeMessagesImpl(channelId,
                    isFilterByParentId,
                    parentId,
                    !includingTags.isEmpty(),
                    includingTags.toArray(new String[0]),
                    excludingTags.toArray(new String[0]),
                    isDeleted);
        } else {
            return observeMessagesWithTypeImpl(channelId,
                    isFilterByParentId,
                    parentId,
                    !includingTags.isEmpty(),
                    includingTags.toArray(new String[0]),
                    excludingTags.toArray(new String[0]),
                    isDeleted,
                    type);
        }
    }

    @Query("SELECT *" +
            " from message" +
            " where message.channelId = :channelId" +
            " and case when :isFilterByParentId and :parentId is not null then message.parentId = :parentId " +
            " when :isFilterByParentId and :parentId is null then message.parentId is null " +
            " else message.messageId is not null end" + // always true
            " and case when :isIncludingTags then message.messageId in (SELECT messageId from message_tag where tagName in (:includingTags))" +
            " else message.messageId is not null end" + // always true
            " and case when :isExcludingTags then message.messageId not in (SELECT messageId from message_tag where tagName in (:excludingTags))" +
            " else message.messageId is not null end" + // always true
            " and case when :isDeleted is not null then message.isDeleted = :isDeleted" +
            " else message.messageId is not null end" + // always true
            " and message.updatedAt > :now" +
            " and message.messageId not in " +
            "(" +
            "SELECT amity_paging_id.id" +
            " from amity_paging_id" +
            " where amity_paging_id.hash = (:hash)" +
            " and amity_paging_id.nonce = (:nonce) " +
            ")" +
            " order by message.updatedAt desc" +
            " limit 1"
    )
    abstract Flowable<EkoMessageEntity> getLatestMessageImpl(String channelId,
                                                             Boolean isFilterByParentId,
                                                             String parentId,
                                                             boolean isIncludingTags,
                                                             String[] includingTags,
                                                             boolean isExcludingTags,
                                                             String[] excludingTags,
                                                             Boolean isDeleted,
                                                             int hash,
                                                             int nonce,
                                                             DateTime now);

    @Query("SELECT *" +
            " from message" +
            " where message.channelId = :channelId" +
            " and case when :isFilterByParentId and :parentId is not null then message.parentId = :parentId " +
            " when :isFilterByParentId and :parentId is null then message.parentId is null " +
            " else message.messageId is not null end" + // always true
            " and case when :isIncludingTags then message.messageId in (SELECT messageId from message_tag where tagName in (:includingTags))" +
            " else message.messageId is not null end" + // always true
            " and case when :isExcludingTags then message.messageId not in (SELECT messageId from message_tag where tagName in (:excludingTags))" +
            " else message.messageId is not null end" + // always true
            " and case when :isDeleted is not null then message.isDeleted = :isDeleted" +
            " else message.messageId is not null end" + // always true
            " and message.updatedAt > :now" +
            " and message.messageId not in " +
            "(" +
            "SELECT amity_paging_id.id" +
            " from amity_paging_id" +
            " where amity_paging_id.hash = (:hash)" +
            " and amity_paging_id.nonce = (:nonce) " +
            ")" +
            " and message.type in (:type)" +
            " order by message.updatedAt desc" +
            " limit 1"
    )
    abstract Flowable<EkoMessageEntity> getLatestMessageWithTypeImpl(String channelId,
                                                             Boolean isFilterByParentId,
                                                             String parentId,
                                                             boolean isIncludingTags,
                                                             String[] includingTags,
                                                             boolean isExcludingTags,
                                                             String[] excludingTags,
                                                             Boolean isDeleted,
                                                             List<String> type,
                                                             int hash,
                                                             int nonce,
                                                             DateTime now);

    public Flowable<EkoMessageEntity> getLatestMessage(String channelId,
                                                       Boolean isFilterByParentId,
                                                       String parentId,
                                                       String[] includingTags,
                                                       String[] excludingTags,
                                                       Boolean isDeleted,
                                                       List<String> type,
                                                       int hash,
                                                       int nonce,
                                                       DateTime now) {
        if (type.isEmpty()) {
            return getLatestMessageImpl(channelId,
                    isFilterByParentId,
                    parentId,
                    includingTags.length > 0,
                    includingTags,
                    excludingTags.length > 0,
                    excludingTags,
                    isDeleted,
                    hash,
                    nonce,
                    now);
        } else {
            return getLatestMessageWithTypeImpl(channelId,
                    isFilterByParentId,
                    parentId,
                    includingTags.length > 0,
                    includingTags,
                    excludingTags.length > 0,
                    excludingTags,
                    isDeleted,
                    type,
                    hash,
                    nonce,
                    now);
        }
    }

}
