package com.ekoapp.ekosdk.internal.data.dao

import androidx.room.*
import com.ekoapp.ekosdk.EkoObject
import org.apache.commons.lang3.NotImplementedException
import org.joda.time.Duration
import org.joda.time.DateTime

abstract class EkoObjectDao<EntityType : EkoObject> {

    open fun getByIdNow(id: String): EntityType? {
        val error = String.format("%s not implemented getByIdNow() yet.", javaClass.name)
        throw NotImplementedException(error)
    }

    open fun getByIdsNow(ids: List<String>): List<EntityType> {
        return emptyList()
    }

    @Insert(onConflict = OnConflictStrategy.REPLACE)
    abstract fun insertImpl(`object`: EntityType)
    @Insert(onConflict = OnConflictStrategy.REPLACE)
    abstract fun insertImpl(objects: List<EntityType>)
    @Update
    abstract fun updateImpl(`object`: EntityType)

    @Transaction
    open fun insert(`object`: EntityType) {
        insertImpl(`object`)
    }

    @Deprecated("insert always replaces. use 'save()' as insertOrUpdate")
    @Transaction
    open fun insert(objects: List<EntityType>) {
        insertImpl(objects)
    }

    @Transaction
    open fun update(`object`: EntityType) {
        updateImpl(`object`)
    }

    open fun update(objects: List<EntityType>) {
        for (`object` in objects) {
            update(`object`)
        }
    }

    @Delete
    abstract fun delete(`object`: EntityType)
    @Delete
    abstract fun delete(objects: List<EntityType>)
    abstract fun deleteAll()

    open fun getExpiration(): Duration = Duration.standardMinutes(1)

    open fun save(
        freshList: List<EntityType>,
    ) {
        val fromNetwork = freshList
        val idsFromNetwork = fromNetwork.map { it.id }
        val onDisk = getByIdsNow(idsFromNetwork)
        val onDiskMap: Map<String, EntityType> = onDisk.associateBy {
            it.id
        }
        val freshObjects: MutableList<EntityType> = mutableListOf()
        val changedObjects: MutableList<EntityType> = mutableListOf()
        for (objectFromNetwork in fromNetwork) {
            val id = objectFromNetwork.id
            val objectOnDisk = onDiskMap[id]
            val expirationDateTime = DateTime.now().plus(getExpiration())
            if (!onDiskMap.containsKey(id)) {
                freshObjects.add(objectFromNetwork.apply {
                    expiresAt = expirationDateTime
                })
            } else if ( objectFromNetwork.updatedAt != null &&
                // add new.updated >= old.updated
                (objectFromNetwork.updatedAt.isAfter(objectOnDisk?.updatedAt)
                || objectFromNetwork.updatedAt.isEqual(objectOnDisk?.updatedAt))
            ) {
                val changedObject = objectFromNetwork.apply {
                    expiresAt = expirationDateTime
                }
                changedObjects.add(changedObject)
            }
        }
        if (freshObjects.isNotEmpty()) {
            insert(freshObjects)
        }
        if (changedObjects.isNotEmpty()) {
            update(changedObjects)
        }
    }
}