package com.ekoapp.ekosdk.internal.data.dao;

import androidx.annotation.Nullable;
import androidx.paging.DataSource;
import androidx.room.Dao;
import androidx.room.Delete;
import androidx.room.Query;

import com.amity.socialcloud.sdk.AmityCoreClient;
import com.ekoapp.ekosdk.internal.data.model.EkoReactionEntity;

import org.joda.time.DateTime;

import java.util.List;

import io.reactivex.rxjava3.core.Flowable;
import io.reactivex.rxjava3.core.Maybe;


@Dao
public abstract class EkoReactionDao extends EkoObjectDao<EkoReactionEntity> {

    @Delete
    public abstract void delete(EkoReactionEntity reaction);

    @Query("DELETE from reaction")
    public abstract void deleteAll();

    @Query("DELETE from reaction" +
            " where referenceType = :referenceType" +
            " and referenceId = :referenceId" +
            " and userId = :userId" +
            " and reactionName = :reactionName" )
    abstract void deleteByPrimaryKeyImpl(String referenceType, String referenceId, String userId, String reactionName);

    public void deleteByPrimaryKey(String referenceType, String referenceId, String userId, String reactionName) {
        deleteByPrimaryKeyImpl(referenceType, referenceId, userId, reactionName);
    }

    @Query("DELETE from reaction where referenceId = :referenceId and userId = :userId and referenceType = :referenceType")
    abstract void deleteByReferenceIdAndUserIdImpl(String referenceId, String referenceType, String userId);

    public void deleteByReferenceIdAndUserId(String referenceId, String referenceType, String userId) {
        deleteByReferenceIdAndUserIdImpl(referenceId, referenceType, userId);
    }

    @Query("DELETE from reaction where referenceId = :referenceId and referenceType = :referenceType")
    abstract void deleteByReferenceIdImpl(String referenceId, String referenceType);

    public void deleteByReferenceId(String referenceId, String referenceType) {
        deleteByReferenceIdImpl(referenceId, referenceType);
    }

    @Query("SELECT *" +
            " from reaction" +
            " where reaction.referenceId = :referenceId" +
            " and reaction.referenceType = :referenceType" +
            " order by reaction.createdAt DESC")
    abstract DataSource.Factory<Integer, EkoReactionEntity> getAllByReferenceIdImpl(String referenceId, String referenceType);

    public DataSource.Factory<Integer, EkoReactionEntity> getAllByReferenceId(String referenceId, String referenceType) {
        return getAllByReferenceIdImpl(referenceId, referenceType);
    }

    @Query("SELECT *" +
            " from reaction" +
            " where reaction.referenceId = :referenceId" +
            " and reaction.reactionName = :reactionName" +
            " and reaction.referenceType = :referenceType" +
            " order by reaction.createdAt DESC")
    abstract DataSource.Factory<Integer, EkoReactionEntity> getAllByReferenceIdAndReactionNameImpl(String referenceId, String referenceType, String reactionName);

    public DataSource.Factory<Integer, EkoReactionEntity> getAllByReferenceIdAndReactionName(String referenceId, String referenceType, String reactionName) {
        return getAllByReferenceIdAndReactionNameImpl(referenceId, referenceType, reactionName);
    }

    @Query("SELECT *" +
            " from reaction" +
            " where reaction.referenceType = :referenceType" +
            " and reaction.referenceId = :referenceId" +
            " and reaction.userId = :userId" +
            " and reaction.reactionName = :reactionName" +
            " LIMIT 1")
    abstract Maybe<EkoReactionEntity> getReactionByReferenceAndReactionNameImpl(String referenceType, String referenceId, String reactionName, String userId);

    public Maybe<EkoReactionEntity> getReactionByReferenceAndReactionName(String referenceType, String referenceId, String reactionName, String userId) {
        return getReactionByReferenceAndReactionNameImpl(referenceType, referenceId, reactionName, userId);
    }

    @Query("SELECT *" +
            " from reaction" +
            " where reaction.referenceType = :referenceType" +
            " and reaction.referenceId = :referenceId" +
            " and reaction.userId = :userId")
    abstract Flowable<List<EkoReactionEntity>> getAllMyReactionsByReferenceTypeAndReferenceIdImpl(String referenceType, String referenceId, String userId);

    public Flowable<List<EkoReactionEntity>> getAllMyReactionsByReferenceTypeAndReferenceId(String referenceType, String referenceId) {
        return getAllMyReactionsByReferenceTypeAndReferenceIdImpl(referenceType, referenceId, AmityCoreClient.INSTANCE.getUserId());
    }

    @Query("SELECT *" +
            " from reaction" +
            " where reaction.referenceType = :referenceType" +
            " and reaction.referenceId = :referenceId" +
            " and reaction.userId = :userId")
    abstract List<EkoReactionEntity> getAllMyReactionsByReferenceTypeAndReferenceIdNowImpl(String referenceType, String referenceId, String userId);

    public List<EkoReactionEntity> getAllMyReactionsByReferenceTypeAndReferenceIdNow(String referenceType, String referenceId) {
        return getAllMyReactionsByReferenceTypeAndReferenceIdNowImpl(referenceType, referenceId, AmityCoreClient.INSTANCE.getUserId());
    }

    @Query("SELECT *" +
            " from reaction" +
            " where reaction.referenceId = :referenceId" +
            " and reaction.referenceType = :referenceType" +
            " order by reaction.createdAt DESC")
    public abstract List<EkoReactionEntity> getAllByReferenceTypeAndReferenceId(String referenceType, String referenceId);


    @Query("SELECT *" +
            " from reaction" +
            " where reaction.reactionId >= :reactionId" +
            " order by reaction.createdAt DESC")
    public abstract List<EkoReactionEntity> getAllAfterReactionId(String reactionId);

    @Query("SELECT *" +
            " from reaction" +
            " where reaction.reactionId < :reactionId" +
            " order by reaction.createdAt DESC")
    public abstract List<EkoReactionEntity> getAllBeforeReactionId(String reactionId);

    @Query("SELECT *" +
            " from reaction" +
            " where reaction.reactionId BETWEEN :fromReactionId and :tillReactionId" +
            " and reaction.reactionId != :tillReactionId" +
            " order by reaction.createdAt DESC")
    public abstract List<EkoReactionEntity> getAllBetweenReactionId(String fromReactionId, String tillReactionId);

    @Query("SELECT *" +
            " from reaction" +
            " where reaction.referenceId = :referenceId" +
            " and reaction.referenceType = :referenceType" +
            " and reaction.createdAt > :now" +
            " order by reaction.createdAt DESC" +
            " limit 1")
    public abstract Flowable<EkoReactionEntity> getLatestReactionWithoutReactionNameImpl(String referenceType, String referenceId, DateTime now);

    @Query("SELECT *" +
            " from reaction" +
            " where reaction.referenceId = :referenceId" +
            " and reaction.referenceType = :referenceType" +
            " and reaction.reactionName = :reactionName" +
            " and reaction.createdAt > :now" +
            " order by reaction.createdAt DESC" +
            " limit 1")
    public abstract Flowable<EkoReactionEntity> getLatestReactionImpl(String referenceType, String referenceId, String reactionName, DateTime now);

    public Flowable<EkoReactionEntity> getLatestReaction(String referenceType, String referenceId, @Nullable String reactionName, DateTime now) {
        if (reactionName == null || reactionName.isEmpty()) {
            return getLatestReactionWithoutReactionNameImpl(referenceType, referenceId, now);
        } else {
            return getLatestReactionImpl(referenceType, referenceId, reactionName, now);
        }
    }

}
