package com.ekoapp.ekosdk.internal.data.dao;

import androidx.paging.DataSource;
import androidx.room.Dao;
import androidx.room.Delete;
import androidx.room.Query;

import com.amity.socialcloud.sdk.social.feed.AmityCommunityFeedSortOption;
import com.amity.socialcloud.sdk.social.feed.AmityFeedType;
import com.amity.socialcloud.sdk.social.feed.AmityUserFeedSortOption;
import com.ekoapp.ekosdk.internal.PostEntity;

import org.joda.time.DateTime;

import java.util.List;

import javax.annotation.Nullable;

import io.reactivex.rxjava3.core.Completable;
import io.reactivex.rxjava3.core.Flowable;

@Dao
public abstract class PostDao extends EkoObjectDao<PostEntity> {

    @Delete
    public abstract void delete(PostEntity ekoPost);

    @Query("DELETE from post")
    public abstract void deleteAll();

    @Query("DELETE from post where parentPostId = :postId or postId = :postId")
    public abstract Completable deleteById(String postId);

    @Query("UPDATE post set commentCount = commentCount + 1 where postId = :postId")
    public abstract void incrementCommentCount(String postId);

    @Query("UPDATE post set commentCount = commentCount - 1 where postId = :postId")
    public abstract void decrementCommentCount(String postId);

    // dummy update post
    @Query("UPDATE post set postId = postId where postId = :postId")
    public abstract void updatePost(String postId);

    @Query("SELECT *" +
            " from post" +
            " where post.postId IN (:postIds)")
    abstract List<PostEntity> getByIdsNowImpl(List<String> postIds);

    public List<PostEntity> getByIdsNow(List<String> postIds) {
        return getByIdsNowImpl(postIds);
    }

    @Query("SELECT *" +
            " from post" +
            " where post.postId IN (:postIds)" +
            " LIMIT 1")
    abstract PostEntity getByIdNowImpl(String postIds);

    public PostEntity getByIdNow(String postId) {
        return getByIdNowImpl(postId);
    }

    @Query("SELECT *" +
            " from post, amity_paging_id" +
            " where post.targetType = :targetType" +
            " and post.targetId = :targetId " +
            " and post.isDeleted =" +
            " (case when :includeDeleted is null then post.isDeleted else :includeDeleted end)" +
            " and post.feedType =" +
            " (case when :feedType is null then post.feedType else :feedType end)" +
            " and post.postDataType in (:postTypes) " +

            " and post.updatedAt > :now" +
            " and post.postId not in " +

            "(" +
            "SELECT amity_paging_id.id" +
            " from amity_paging_id" +
            " where amity_paging_id.hash = (:hash)" +
            " and amity_paging_id.nonce = (:nonce) " +
            ")" +

            " order by post.updatedAt  desc" +
            " limit 1"
    )
    abstract Flowable<PostEntity> getLatestChildrenPostImpl(String targetId,
                                                            String targetType,
                                                            List<String> postTypes,
                                                            int hash,
                                                            int nonce,
                                                            DateTime now,
                                                            @Nullable Boolean includeDeleted,
                                                            @Nullable String feedType);


    @Query("SELECT *" +
            " from post" +
            " where post.targetType = :targetType" +
            " and post.targetId = :targetId " +
            " and post.parentPostId is null" +

            " and post.isDeleted =" +
            " (case when :includeDeleted is null then post.isDeleted else :includeDeleted end)" +
            " and post.feedType =" +
            " (case when :feedType is null then post.feedType else :feedType end)" +

            " and post.updatedAt > :now" +
            " and post.postId not in " +

            "(" +
            "SELECT amity_paging_id.id" +
            " from amity_paging_id" +
            " where amity_paging_id.hash = (:hash)" +
            " and amity_paging_id.nonce = (:nonce) " +
            ")" +

            " order by post.updatedAt  desc" +
            " limit 1"
    )
    abstract Flowable<PostEntity> getLatestParentPostImpl(String targetId,
                                                          String targetType,
                                                          int hash,
                                                          int nonce,
                                                          DateTime now,
                                                          @Nullable Boolean includeDeleted,
                                                          @Nullable String feedType);


    public Flowable<PostEntity> getLatestPost(String targetId,
                                              String targetType,
                                              Boolean includeDeleted,
                                              List<String> postTypes,
                                              int hash,
                                              int nonce,
                                              @Nullable AmityFeedType feedType) {
        @Nullable String feedTypeKey = feedType != null ? feedType.getApiKey() : null;
        if (postTypes == null || postTypes.isEmpty()) {
            return getLatestParentPostImpl(targetId, targetType, hash, nonce,
                    DateTime.now(), includeDeleted, feedTypeKey);
        } else {
            return getLatestChildrenPostImpl(targetId, targetType, postTypes,
                    hash, nonce, DateTime.now(), includeDeleted, feedTypeKey);
        }
    }

    @Query("UPDATE post SET isDeleted = :isDeleted WHERE postId LIKE :postId ")
    abstract Completable updateDeletedImpl(String postId, boolean isDeleted);

    public Completable softDeleteById(String postId) {
        return updateDeletedImpl(postId, true);
    }

    @Query("UPDATE post SET feedType = :feedType WHERE postId = :postId ")
    abstract void updateFeedTypeImpl(String postId, String feedType);

    public void updateFeedType(String postId, String feedType) {
        updateFeedTypeImpl(postId, feedType);
    }

    @Query("SELECT *" +
            " from post" +
            " where post.postId IN (:postIds)" +
            " LIMIT 1")
    abstract io.reactivex.rxjava3.core.Flowable<PostEntity> getPostImpl(String postIds);

    public io.reactivex.rxjava3.core.Flowable<PostEntity> getPost(String postId) {
        return getPostImpl(postId);
    }

    @Query("SELECT *" +
            " from post" +
            " where post.postId IN (:postId)")
    abstract io.reactivex.rxjava3.core.Flowable<List<PostEntity>> observePostImpl(String postId);

    public io.reactivex.rxjava3.core.Flowable<List<PostEntity>> observePost(String postId) {
        return observePostImpl(postId);
    }


    public Flowable<List<PostEntity>> getByPostIds(List<String> ids) {
        return getByPostIdsImpl(ids);
    }

    @Query("SELECT *" +
            " from post" +
            " where post.postId IN (:postIds)")
    abstract Flowable<List<PostEntity>> getByPostIdsImpl(List<String> postIds);

    @Query("SELECT *" +
            " from post, community_feed_query_token" +
            " where post.targetType = 'community'" +
            " and post.targetId = :communityId " +
            " and post.feedType = :feedType" +
            " and post.parentPostId is null" +
            " and post.isDeleted =" +
            " (case when :isDeleted is null then post.isDeleted else :isDeleted end)" +
            " and" +
            " (community_feed_query_token.communityId = :communityId" +
            " and community_feed_query_token.feedType = :feedType" +
            " and community_feed_query_token.isDeleted = :deletedStatus" +
            " and community_feed_query_token.sortBy = :sortBy" +
            " and community_feed_query_token.pageNumber = 1" +
            " and community_feed_query_token.ids like '%' || post.postId || '%')" +
            " order by" +
            " case when :isSortedAsc = 1 then post.createdAt end asc," +
            " case when :isSortedAsc = 0 then post.createdAt end desc"
    )
    abstract DataSource.Factory<Integer, PostEntity> getCommunityFeedDataSourceImpl(
            String communityId,
            String feedType,
            Boolean isDeleted,
            String deletedStatus,
            Boolean isSortedAsc,
            String sortBy);

    public DataSource.Factory<Integer, PostEntity> getCommunityFeedDataSource(String communityId,
                                                                              String feedType,
                                                                              Boolean isDeleted,
                                                                              AmityCommunityFeedSortOption sortOption) {
        String deletedStatus = "";
        if (isDeleted != null) {
            deletedStatus = isDeleted.toString();
        }
        return getCommunityFeedDataSourceImpl(communityId, feedType, isDeleted, deletedStatus, sortOption != AmityCommunityFeedSortOption.LAST_CREATED, sortOption.getApiKey());
    }


    @Query("SELECT *" +
            " from post, user_feed_query_token" +
            " where post.targetType = 'user'" +
            " and post.targetId = :userId " +
            " and post.parentPostId is null" +
            " and post.isDeleted =" +
            " (case when :isDeleted is null then post.isDeleted else :isDeleted end)" +
            " and" +
            " (user_feed_query_token.userId = :userId" +
            " and user_feed_query_token.isDeleted = :deletedStatus" +
            " and user_feed_query_token.sortBy = :sortBy" +
            " and user_feed_query_token.pageNumber = 1" +
            " and user_feed_query_token.ids like '%' || post.postId || '%')" +
            " order by" +
            " case when :isSortedAsc = 1 then post.createdAt end asc," +
            " case when :isSortedAsc = 0 then post.createdAt end desc"
    )
    abstract DataSource.Factory<Integer, PostEntity> getUserFeedDataSourceImpl(String userId,
                                                                               Boolean isDeleted,
                                                                               String deletedStatus,
                                                                               Boolean isSortedAsc,
                                                                               String sortBy);

    public DataSource.Factory<Integer, PostEntity> getUserFeedDataSource(String userId,
                                                                         Boolean isDeleted,
                                                                         AmityUserFeedSortOption sortOption) {
        String deletedStatus = "";
        if (isDeleted != null) {
            deletedStatus = isDeleted.toString();
        }
        return getUserFeedDataSourceImpl(userId, isDeleted, deletedStatus, sortOption == AmityUserFeedSortOption.FIRST_CREATED, sortOption.getApiKey());
    }

    @Query("SELECT *" +
            " from post, global_post" +
            " where post.isDeleted = 0" +
            " and post.postId = global_post.postId"
    )
    abstract DataSource.Factory<Integer, PostEntity> getGlobalPostDataSourceImpl();

    public DataSource.Factory<Integer, PostEntity> getGlobalPostDataSource() {
        return getGlobalPostDataSourceImpl();
    }
}