package com.ekoapp.ekosdk.internal.push;

import android.util.Pair;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.amity.socialcloud.sdk.log.AmityLog;
import com.amity.socialcloud.sdk.push.EkoPushContract;
import com.ekoapp.ekosdk.internal.api.EkoApi;
import com.ekoapp.ekosdk.internal.api.http.request.RegisterDeviceForPushNotificationRequest;
import com.ekoapp.ekosdk.internal.api.http.request.UnregisterDeviceForPushNotificationRequest;
import com.ekoapp.ekosdk.internal.data.EkoDatabase;
import com.ekoapp.ekosdk.internal.data.model.EkoAccount;
import com.ekoapp.ekosdk.internal.data.model.EkoApiKey;
import com.ekoapp.ekosdk.internal.data.model.EkoBaiduToken;
import com.ekoapp.ekosdk.internal.data.model.EkoFcmToken;
import com.ekoapp.ekosdk.internal.data.model.EkoPushConfig;
import com.github.davidmoten.guavamini.Objects;
import com.google.gson.Gson;

import io.reactivex.Completable;
import io.reactivex.Flowable;
import io.reactivex.schedulers.Schedulers;

public class EkoPushContractImpl implements EkoPushContract {
    private final String TAG = getClass().getName();

    enum Provider {
        FCM("fcm"),
        BAIDU("baidu");

        private final String apiKey;


        Provider(String apiKey) {
            this.apiKey = apiKey;
        }

        public String getApiKey() {
            return apiKey;
        }
    }

    /**
     * comment
     */
    public EkoPushContractImpl() {
        initPushService();
    }

    private void initPushService() {
        Flowable.combineLatest(
                initFcm().startWith(new TokenConfigAndApiKey<EkoFcmToken>(null, null, null)),
                initBaidu().startWith(new TokenConfigAndApiKey<EkoBaiduToken>(null, null, null)),
                (fcmConfig, baiduConfig) -> new Pair<>(fcmConfig, baiduConfig)
        ).skip(1)
                .flatMapCompletable((configPair) -> {
                    TokenConfigAndApiKey<EkoFcmToken> fcmConfig = configPair.first;
                    String fcmState = "null";
                    if(fcmConfig.config != null) {
                        fcmState = fcmConfig.config.getState().getApiKey();
                    }
                    String fcmToken = "null";
                    if(fcmConfig.token != null) {
                        fcmToken = fcmConfig.token.getToken();
                    }
                    AmityLog.INSTANCE.e( "fcm: " + fcmState + " token:" + fcmToken);

                    TokenConfigAndApiKey<EkoBaiduToken> baiduConfig = configPair.second;
                    String baiduState = "null";
                    if(baiduConfig.config != null) {
                        baiduState = baiduConfig.config.getState().getApiKey();
                    }
                    String baiduToken = "null";
                    if(baiduConfig.token != null) {
                        baiduToken = baiduConfig.token.getToken();
                    }
                    AmityLog.INSTANCE.e( "baidu: " + baiduState + " token:" + baiduToken);

                    if (fcmConfig.token != null && baiduConfig.token != null) {
                        return prioritizeFcmToken(fcmConfig, baiduConfig);
                    } else if (fcmConfig.token != null) {
                        return handleFcmToken(fcmConfig);
                    } else {
                        return handleBaiduToken(baiduConfig);
                    }
                })
                .subscribeOn(Schedulers.io())
                .subscribe();
    }

    private static Flowable<EkoPushConfig> PUSH_CONFIG = EkoDatabase.get()
            .accountDao()
            .getAll()
            .flatMapIterable(accounts -> accounts)
            .distinct(EkoAccount::getUserId)
            .flatMap(account -> EkoDatabase.get()
                    .pushConfigDao()
                    .getPushConfig(account.getUserId())
                    .distinctUntilChanged(config -> config));

    private static Flowable<EkoApiKey> API_KEY = EkoDatabase.get()
            .apiKeyDao()
            .getCurrentApiKeyFlowable();

    private Flowable<TokenConfigAndApiKey<EkoFcmToken>> initFcm() {
        return Flowable.combineLatest(EkoDatabase.get()
                .fcmTokenDao()
                .getFcmToken()
                .distinctUntilChanged(fcmToken -> fcmToken), PUSH_CONFIG, API_KEY, TokenConfigAndApiKey::new);
    }

    private Flowable<TokenConfigAndApiKey<EkoBaiduToken>> initBaidu() {
        return Flowable.combineLatest(EkoDatabase.get()
                .baiduTokenDao()
                .getBaiduToken()
                .distinctUntilChanged(baiduToken -> baiduToken), PUSH_CONFIG, API_KEY, TokenConfigAndApiKey::new)
                .filter(tokenConfigAndApiKey -> tokenConfigAndApiKey.token.getUserId() != null && tokenConfigAndApiKey.token.getChannelId() != null);
    }

    private Completable prioritizeFcmToken(TokenConfigAndApiKey<EkoFcmToken> fcmConfig, TokenConfigAndApiKey<EkoBaiduToken> baiduConfig) {
        if (Objects.equal(EkoPushConfig.State.REGISTERED, fcmConfig.config.getState())) {
            RegisterDeviceForPushNotificationRequest registerRequest = RegisterDeviceForPushNotificationRequest.create(fcmConfig.config.getUserId(),
                    fcmConfig.config.getDeviceId(),
                    fcmConfig.token.getToken(),
                    Provider.FCM.getApiKey());

            UnregisterDeviceForPushNotificationRequest unregisterRequest = UnregisterDeviceForPushNotificationRequest.create(baiduConfig.config.getUserId(),
                    baiduConfig.config.getDeviceId());

            return EkoApi.Companion.notification()
                    .unregisterNotificationToken(baiduConfig.apiKey.getApiKey(), unregisterRequest)
                    .ignoreElement()
                    .andThen(
                            EkoApi.Companion.notification()
                                    .registerNotificationToken(fcmConfig.apiKey.getApiKey(), registerRequest)
                                    .doOnSuccess(response -> AmityLog.INSTANCE.tag(TAG).i(String.format("fcm response:%s", new Gson().toJson(response))))
                                    .ignoreElement()
                    )
                    .onErrorComplete();
        } else {
            UnregisterDeviceForPushNotificationRequest request = UnregisterDeviceForPushNotificationRequest.create(fcmConfig.config.getUserId(),
                    fcmConfig.config.getDeviceId());

            AmityLog.INSTANCE.tag(TAG).i(String.format("fcm un-registration request:%s", new Gson().toJson(request)));

            return EkoApi.Companion.notification()
                    .unregisterNotificationToken(fcmConfig.apiKey.getApiKey(), request)
                    .doOnSuccess(response -> AmityLog.INSTANCE.tag(TAG).i(String.format("fcm response:%s", new Gson().toJson(response))))
                    .ignoreElement()
                    .onErrorComplete();
        }
    }

    private Completable handleFcmToken(TokenConfigAndApiKey<EkoFcmToken> tokenConfigAndApiKey) {
        if (Objects.equal(EkoPushConfig.State.REGISTERED, tokenConfigAndApiKey.config.getState())) {
            RegisterDeviceForPushNotificationRequest request = RegisterDeviceForPushNotificationRequest.create(tokenConfigAndApiKey.config.getUserId(),
                    tokenConfigAndApiKey.config.getDeviceId(),
                    tokenConfigAndApiKey.token.getToken(),
                    Provider.FCM.getApiKey());

            AmityLog.INSTANCE.tag(TAG).i(String.format("fcm registration request:%s", new Gson().toJson(request)));

            return EkoApi.Companion.notification()
                    .registerNotificationToken(tokenConfigAndApiKey.apiKey.getApiKey(), request)
                    .doOnSuccess(response -> AmityLog.INSTANCE.tag(TAG).i(String.format("fcm response:%s", new Gson().toJson(response))))
                    .ignoreElement()
                    .onErrorComplete();
        } else {
            String userId = tokenConfigAndApiKey.config.getUserId();
            UnregisterDeviceForPushNotificationRequest request = UnregisterDeviceForPushNotificationRequest.create(tokenConfigAndApiKey.config.getUserId(),
                    tokenConfigAndApiKey.config.getDeviceId());

            AmityLog.INSTANCE.tag(TAG).i(String.format("fcm un-registration request:%s", new Gson().toJson(request)));

            return EkoApi.Companion.notification()
                    .unregisterNotificationToken(tokenConfigAndApiKey.apiKey.getApiKey(), request)
                    .doOnSuccess(response -> AmityLog.INSTANCE.tag(TAG).i(String.format("fcm response:%s", new Gson().toJson(response))))
                    .ignoreElement()
                    .andThen(Completable.fromAction(() -> EkoDatabase.get().pushConfigDao().clearUnregisteredPushConfigForUser(userId))) // Remove unregister success config from DB
                    .onErrorComplete();
        }
    }

    private Completable handleBaiduToken(TokenConfigAndApiKey<EkoBaiduToken> tokenConfigAndApiKey) {
        if (Objects.equal(EkoPushConfig.State.REGISTERED, tokenConfigAndApiKey.config.getState())) {
            RegisterDeviceForPushNotificationRequest request = RegisterDeviceForPushNotificationRequest.create(tokenConfigAndApiKey.config.getUserId(),
                    tokenConfigAndApiKey.config.getDeviceId(),
                    new Gson().toJson(new RegisterDeviceForPushNotificationRequest.BaiduToken(tokenConfigAndApiKey.token.getToken(), tokenConfigAndApiKey.token.getUserId(), tokenConfigAndApiKey.token.getChannelId())),
                    Provider.BAIDU.apiKey);

            AmityLog.INSTANCE.tag(TAG).i(String.format("baidu registration request:%s", new Gson().toJson(request)));

            return EkoApi.Companion.notification()
                    .registerNotificationToken(tokenConfigAndApiKey.apiKey.getApiKey(), request)
                    .doOnSuccess(response -> AmityLog.INSTANCE.tag(TAG).i(String.format("baidu response:%s", new Gson().toJson(response))))
                    .ignoreElement()
                    .onErrorComplete();
        } else {
            UnregisterDeviceForPushNotificationRequest request = UnregisterDeviceForPushNotificationRequest.create(tokenConfigAndApiKey.config.getUserId(),
                    tokenConfigAndApiKey.config.getDeviceId());

            AmityLog.INSTANCE.tag(TAG).i(String.format("baidu un-registration request:%s", new Gson().toJson(request)));

            return EkoApi.Companion.notification()
                    .unregisterNotificationToken(tokenConfigAndApiKey.apiKey.getApiKey(), request)
                    .doOnSuccess(response -> AmityLog.INSTANCE.tag(TAG).i(String.format("baidu response:%s", new Gson().toJson(response))))
                    .ignoreElement()
                    .onErrorComplete();
        }
}

    @Override
    public Completable insertBaiduApiKey(@NonNull String apiKey) {
        return Completable.fromAction(() -> EkoDatabase.get()
                .baiduTokenDao()
                .insertOrUpdate(new EkoBaiduToken(apiKey)));
    }

    @Override
    public Completable updateBaiduToken(@Nullable String token, @Nullable String userId, @Nullable String channelId) {
        return Completable.fromAction(() -> EkoDatabase.get()
                .baiduTokenDao()
                .updateToken(token, userId, channelId));
    }

    @Override
    public Completable insertFcmToken(@NonNull String fcmToken) {
        return Completable.fromAction(() -> EkoDatabase.get().fcmTokenDao().insert(new EkoFcmToken(fcmToken)));
    }

    @Override
    public Flowable<Boolean> hasRegisteredConfig() {
        return EkoDatabase.get()
                .pushConfigDao()
                .hasRegisteredConfig();
    }

    class TokenConfigAndApiKey<TOKEN> {

        TOKEN token;
        EkoPushConfig config;
        EkoApiKey apiKey;


        TokenConfigAndApiKey(TOKEN token, EkoPushConfig config, EkoApiKey apiKey) {
            this.token = token;
            this.config = config;
            this.apiKey = apiKey;
        }
    }

}
