package com.ekoapp.ekosdk.internal.repository.comment

import androidx.paging.ExperimentalPagingApi
import com.amity.socialcloud.sdk.social.comment.AmityComment
import com.amity.socialcloud.sdk.social.comment.AmityCommentSortOption
import com.amity.socialcloud.sdk.social.data.comment.CommentQueryPersister
import com.amity.socialcloud.sdk.social.data.comment.CommentRemoteDataStore
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.internal.api.dto.EkoCommentAndUserListDto
import com.ekoapp.ekosdk.internal.api.socket.request.CommentQueryRequest
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.entity.CommentEntity
import com.ekoapp.ekosdk.internal.keycreator.CommentKeyCreator
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import com.ekoapp.ekosdk.internal.token.QueryStreamQueryToken
import com.ekoapp.ekosdk.internal.usecase.comment.CommentReactorUseCase
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single

@OptIn(ExperimentalPagingApi::class)
class CommentMediator(
    val referenceId: String,
    val referenceType: String,
    val isFilterByParentId: Boolean,
    val parentId: String?,
    val isDeleted: Boolean?,
    val sortOption: AmityCommentSortOption
) :
    DynamicQueryStreamMediator<CommentEntity, EkoCommentAndUserListDto, AmityComment>(
        dynamicQueryStreamKeyCreator = CommentKeyCreator(
            referenceId = referenceId,
            referenceType = referenceType,
            parentId = parentId,
            isDeleted = isDeleted,
            sortOption = sortOption
        ),
        nonce = AmityNonce.COMMENT_LIST
    ) {
    override fun provideReactorPublisher(): Flowable<AmityComment> {
        return CommentReactorUseCase()
            .execute(
                referenceId = referenceId,
                referenceType = referenceType,
                parentId = parentId,
                isDeleted = isDeleted,
                dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
                nonce = AmityNonce.COMMENT_LIST
            )
    }

    override fun getFirstPageRequest(pageSize: Int): Single<EkoCommentAndUserListDto> {
        val requestOption: CommentQueryRequest.CommentQueryRequestOptions =
            if (sortOption == AmityCommentSortOption.FIRST_CREATED) {
                CommentQueryRequest.CommentQueryRequestOptions(first = pageSize, type = scrollable)
            } else {
                CommentQueryRequest.CommentQueryRequestOptions(last = pageSize, type = scrollable)
            }
        return getRequest(requestOption)
    }

    override fun getFetchByTokenRequest(token: String): Single<EkoCommentAndUserListDto> {
        return getRequest(
            CommentQueryRequest.CommentQueryRequestOptions(
                token = token,
                type = scrollable
            )
        )
    }

    override fun persistResponse(dto: EkoCommentAndUserListDto): Completable {
        return CommentQueryPersister().persist(dto).toV2()
    }

    override fun convertResponseToQueryToken(dto: EkoCommentAndUserListDto): QueryStreamQueryToken {
        val isFirsCreated = sortOption == AmityCommentSortOption.FIRST_CREATED
        return DynamicQueryStreamQueryToken(
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            next = if (isFirsCreated) dto.token.next else dto.token.previous,
            previous = if (isFirsCreated) dto.token.previous else dto.token.next,
            primaryKeys = dto.comments?.map { it.commentId } ?: emptyList()
        )
    }

    private fun getRequest(options: CommentQueryRequest.CommentQueryRequestOptions): Single<EkoCommentAndUserListDto> {
        return CommentRemoteDataStore().getComments(
            CommentQueryRequest(
                referenceId = referenceId,
                referenceType = referenceType,
                parentId = parentId,
                isDeleted = isDeleted,
                filterByParentId = isFilterByParentId,
                options = options
            )
        ).toV2()
    }
}

private const val scrollable = "scrollable"