package com.ekoapp.ekosdk.internal.repository.community

import androidx.paging.ExperimentalPagingApi
import com.amity.socialcloud.sdk.AmityCoreClient
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.amity.socialcloud.sdk.social.community.AmityCommunityCategorySortOption
import com.amity.socialcloud.sdk.social.data.category.CategoryQueryPersister
import com.amity.socialcloud.sdk.social.data.category.CategoryRemoteDataStore
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.community.category.CommunityCategoryEntity
import com.ekoapp.ekosdk.internal.AmityRxRemoteMediator
import com.ekoapp.ekosdk.internal.api.socket.request.CommunityCategoriesQueryRequest
import com.ekoapp.ekosdk.internal.data.dao.AmityCommunityCategoryQueryTokenDao
import com.ekoapp.ekosdk.internal.data.model.EkoCommunityCategoryQueryToken
import io.reactivex.Maybe
import io.reactivex.Single

@OptIn(ExperimentalPagingApi::class)
internal class CommunityCategoryRxRemoteMediator(
    private val sortOption: AmityCommunityCategorySortOption,
    private val isDeleted: Boolean?,
    communityCategoryQueryTokenDao: AmityCommunityCategoryQueryTokenDao
) : AmityRxRemoteMediator<CommunityCategoryEntity,
        EkoCommunityCategoryQueryToken,
        AmityCommunityCategoryQueryTokenDao>(communityCategoryQueryTokenDao) {

    override fun loadPage(pageNumber: Int, pageSize: Int): Maybe<EkoCommunityCategoryQueryToken> {
        val options =
            CommunityCategoriesQueryRequest.CommunityCategoriesQueryOptions(limit = pageSize)
        return executeRequest(options = options, pageNumber = pageNumber)
    }

    override fun loadFirstPage(pageSize: Int): Maybe<EkoCommunityCategoryQueryToken> {
        val options =
            CommunityCategoriesQueryRequest.CommunityCategoriesQueryOptions(limit = pageSize)
        return executeRequest(options = options, pageNumber = 1)
    }

    override fun loadNextPage(
        token: EkoCommunityCategoryQueryToken,
        pageSize: Int
    ): Maybe<EkoCommunityCategoryQueryToken> {
        val options =
            CommunityCategoriesQueryRequest.CommunityCategoriesQueryOptions(token = token.next)
        return executeRequest(options = options, pageNumber = token.pageNumber + 1)
    }

    private fun executeRequest(
        options: CommunityCategoriesQueryRequest.CommunityCategoriesQueryOptions,
        pageNumber: Int
    ): Maybe<EkoCommunityCategoryQueryToken> {
        return CategoryRemoteDataStore().queryCategories(
            sortBy = sortOption.apiKey,
            isDeleted = isDeleted,
            options = QueryOptionsRequestParams(limit = options.limit, token = options.token)
        ).toV2()
            .flatMap { dto ->
                CategoryQueryPersister().persist(dto).toV2()
                    .andThen(Single.just(dto))
            }
            .map {
                EkoCommunityCategoryQueryToken().apply {
                    this.updatedAt = AmityCoreClient.getServerTime()
                    this.pageNumber = pageNumber
                    this.next = it.token?.next
                    this.previous = it.token?.previous
                    this.ids = it.results.map { it.categoryId }
                }
            }
            .toMaybe()
    }

    override fun primaryKeys(): Map<String, Any> {
        return mapOf(
            "sortBy" to sortOption.apiKey,
            "isDeleted" to (isDeleted ?: false)
        )
    }

    override fun applyPrimaryKeysToToken(token: EkoCommunityCategoryQueryToken): EkoCommunityCategoryQueryToken {
        return token.apply {
            this.sortBy = this@CommunityCategoryRxRemoteMediator.sortOption.apiKey
            this.isDeleted = this@CommunityCategoryRxRemoteMediator.isDeleted ?: false
        }
    }

    override fun stackFromEnd(): Boolean {
        return false
    }

}