package com.ekoapp.ekosdk.internal.repository.community

import androidx.paging.ExperimentalPagingApi
import com.amity.socialcloud.sdk.core.permission.AmityRoles
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.amity.socialcloud.sdk.social.community.AmityCommunityMembershipSortOption
import com.amity.socialcloud.sdk.social.data.community.CommunityQueryPersister
import com.amity.socialcloud.sdk.social.data.community.membership.CommunityMembershipRemoteDataStore
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.community.membership.query.AmityCommunityMembership
import com.ekoapp.ekosdk.internal.api.dto.CommunityListQueryDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.entity.CommunityMembershipEntity
import com.ekoapp.ekosdk.internal.keycreator.CommunityMembershipKeyCreator
import com.ekoapp.ekosdk.internal.mediator.QueryStreamMediator
import com.ekoapp.ekosdk.internal.token.QueryStreamQueryToken
import io.reactivex.Completable
import io.reactivex.Single

@ExperimentalPagingApi
class CommunityMembershipMediator(
    private val communityId: String,
    private val roles: AmityRoles,
    private val filters: List<AmityCommunityMembership>?,
    private val sortBy: AmityCommunityMembershipSortOption? = null,
    private val keyword: String? = null
) : QueryStreamMediator<CommunityMembershipEntity, CommunityListQueryDto>(
    nonce = AmityNonce.COMMUNITY_MEMBERSHIP,
    queryStreamKeyCreator = CommunityMembershipKeyCreator(
        communityId,
        roles,
        filters,
        sortBy?.apiKey,
        keyword
    ),
) {

    override fun forceRefresh(): Boolean = true

    override fun getFirstPageRequest(pageSize: Int): Single<CommunityListQueryDto> {
        val options = QueryOptionsRequestParams(limit = pageSize)
        return CommunityMembershipRemoteDataStore().queryCommunityUsers(
            communityId,
            filters?.map { it.value },
            roles,
            keyword,
            sortBy?.apiKey,
            options
        ).toV2()
    }

    override fun getFetchByTokenRequest(token: String): Single<CommunityListQueryDto> {
        val options = QueryOptionsRequestParams(token = token)
        return CommunityMembershipRemoteDataStore().queryCommunityUsers(
            communityId,
            filters?.map { it.value },
            roles,
            keyword,
            sortBy?.apiKey,
            options
        ).toV2()
    }

    override fun persistResponse(dto: CommunityListQueryDto): Completable {
        return CommunityQueryPersister().persist(dto).toV2()
    }

    override fun convertResponseToQueryToken(dto: CommunityListQueryDto): QueryStreamQueryToken {
        return QueryStreamQueryToken(
            queryStreamKeyCreator = CommunityMembershipKeyCreator(
                communityId,
                roles,
                filters,
                sortBy?.apiKey,
                keyword
            ),
            next = dto.paging?.next,
            previous = dto.paging?.previous,
            primaryKeys = dto.communityUsers.map { user -> user.userId ?: "" }
        )
    }
}