package com.ekoapp.ekosdk.internal.repository.community

import androidx.paging.ExperimentalPagingApi
import co.amity.rxremotemediator.AmityQueryTokenDao
import co.amity.rxremotemediator.PageKeyedRxRemoteMediator
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.amity.socialcloud.sdk.social.community.AmityCommunityFilter
import com.amity.socialcloud.sdk.social.community.AmityCommunitySortOption
import com.amity.socialcloud.sdk.social.data.community.CommunityQueryPersister
import com.amity.socialcloud.sdk.social.data.community.CommunityRemoteDataStore
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.internal.api.dto.CommunityListQueryDto
import com.ekoapp.ekosdk.internal.api.socket.request.CommunityQueryRequest
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.data.model.CommunityQueryToken
import com.ekoapp.ekosdk.internal.entity.CommunityEntity
import io.reactivex.Single

@OptIn(ExperimentalPagingApi::class)
internal class CommunityPageKeyedRxRemoteMediator(
    private val keyword: String,
    private val categoryId: String,
    private val filter: AmityCommunityFilter,
    private val sortBy: AmityCommunitySortOption,
    private val isDeleted: Boolean?,
    tokenDao: AmityQueryTokenDao
) : PageKeyedRxRemoteMediator<CommunityEntity, CommunityQueryToken>(
        nonce = AmityNonce.COMMUNITY_LIST,
        queryParameters = mapOf(
                "keyword" to keyword,
                "categoryId" to categoryId,
                "filter" to filter.value,
                "sortBy" to sortBy.apiKey,
                "isDeleted" to (isDeleted ?: false)
        ),
        tokenDao = tokenDao
) {
    
    override fun forceRefresh() = true
    
    override fun fetchByToken(token: String): Single<CommunityQueryToken> {
        return fetchDataByToken(token)
                .map(::toQueryToken)
    }
    
    override fun fetchFirstPage(pageSize: Int): Single<CommunityQueryToken> {
        return getRequest(CommunityQueryRequest.CommunityQueryOptions(limit = pageSize))
                .let(::call)
                .map(::toQueryToken)
    }
    
    private fun fetchDataByToken(token: String): Single<CommunityListQueryDto> {
        return getRequest(CommunityQueryRequest.CommunityQueryOptions(token = token)).let(::call)
    }
    
    private fun getRequest(options: CommunityQueryRequest.CommunityQueryOptions): CommunityQueryRequest {
        return CommunityQueryRequest(
                keyword = keyword,
                categoryId = categoryId.ifEmpty { null },
                filter = filter.value,
                sortBy = sortBy.apiKey,
                isDeleted = isDeleted,
                options = options
        )
    }
    
    private fun call(request: CommunityQueryRequest): Single<CommunityListQueryDto> {
        return CommunityRemoteDataStore().queryCommunities(
            keyword = keyword.ifEmpty { null },
            categoryId = categoryId.ifEmpty { null },
            filter = filter.value,
            sortBy = sortBy.apiKey,
            isDeleted = isDeleted,
            options = QueryOptionsRequestParams(
                limit = request.options.limit,
                token = request.options.token
            )
        ).toV2()
                .flatMap {
                    CommunityQueryPersister().persist(it).toV2().andThen(Single.just(it))
                }
    }
    
    private fun toQueryToken(dto: CommunityListQueryDto): CommunityQueryToken {
        return CommunityQueryToken(
                        keyword = keyword,
                        categoryId = categoryId,
                        filter = filter.value,
                        sortBy = sortBy.apiKey,
                        isDeleted = (isDeleted ?: false),
                        next = dto.paging?.next,
                        previous = dto.paging?.previous,
                        ids = dto.communities.mapNotNull { it.communityId }
                )
    }

}