package com.ekoapp.ekosdk.internal.repository.community

import androidx.paging.ExperimentalPagingApi
import co.amity.rxremotemediator.AmityQueryTokenDao
import co.amity.rxremotemediator.PageKeyedRxRemoteMediator
import com.amity.socialcloud.sdk.core.permission.AmityRoles
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.amity.socialcloud.sdk.social.data.community.CommunityQueryPersister
import com.amity.socialcloud.sdk.social.data.community.membership.CommunityMembershipRemoteDataStore
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.community.membership.query.AmityCommunityMembership
import com.ekoapp.ekosdk.internal.api.dto.CommunityListQueryDto
import com.ekoapp.ekosdk.internal.api.socket.request.CommunityMembershipQueryRequest
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.data.model.CommunityMembershipSearchQueryToken
import com.ekoapp.ekosdk.internal.entity.CommunityMembershipEntity
import io.reactivex.Single

@OptIn(ExperimentalPagingApi::class)
internal class SearchMembershipRxRemoteMediator(
    private val communityId: String,
    private val roles: AmityRoles,
    private val filters: List<AmityCommunityMembership>?,
    private val keyword: String?,
    tokenDao: AmityQueryTokenDao
) : PageKeyedRxRemoteMediator<CommunityMembershipEntity, CommunityMembershipSearchQueryToken>(
    nonce = AmityNonce.COMMUNITY_MEMBERSHIP,
    queryParameters = mapOf(
        "communityId" to communityId,
        "roles" to roles,
        "memberships" to (filters ?: listOf()),
        "search" to (keyword ?: "")
    ), tokenDao = tokenDao
) {

    override fun fetchByToken(token: String): Single<CommunityMembershipSearchQueryToken> {
        return fetchDataByToken(token)
            .flatMap { communityDto ->
                Single.just(
                    CommunityMembershipSearchQueryToken(
                        communityId = communityId,
                        roles = roles,
                        filters = filters,
                        keyword = keyword,
                        next = communityDto.paging?.next,
                        previous = communityDto.paging?.previous,
                        ids = communityDto.communityUsers?.map { it.userId ?: "" } ?: listOf()
                    )
                )
            }
    }

    override fun fetchFirstPage(pageSize: Int): Single<CommunityMembershipSearchQueryToken> {
        val options =
            CommunityMembershipQueryRequest.CommunityMembershipQueryOptions(limit = pageSize)
        val request = CommunityMembershipQueryRequest(
            communityId = communityId,
            roles = roles.takeIf { it.isNotEmpty() },
            memberships = filters?.map { it.value },
            options = options,
            search = keyword
        )
        return call(request)
            .flatMap { communityDto ->
                Single.just(
                    CommunityMembershipSearchQueryToken(
                        communityId = communityId,
                        roles = roles,
                        filters = filters,
                        keyword = keyword,
                        next = communityDto.paging?.next,
                        previous = null,
                        ids = communityDto.communityUsers?.map { it.userId ?: "" } ?: listOf()
                    )
                )
            }
    }

    private fun fetchDataByToken(token: String): Single<CommunityListQueryDto> {
        val options = CommunityMembershipQueryRequest.CommunityMembershipQueryOptions(token = token)
        val request = CommunityMembershipQueryRequest(
            communityId = communityId,
            roles = roles.takeIf { it.isNotEmpty() },
            memberships = filters?.map { it.value },
            options = options,
            search = keyword
        )
        return call(request)
    }

    private fun call(request: CommunityMembershipQueryRequest): Single<CommunityListQueryDto> {
        return CommunityMembershipRemoteDataStore().searchCommunityUsers(
            communityId = communityId,
            keyword = keyword,
            roles = roles.takeIf { it.isNotEmpty() },
            memberships = filters?.map { it.value },
            options = QueryOptionsRequestParams(
                limit = request.options?.limit,
                token = request.options?.token
            )
        ).toV2()
            .flatMap {
                CommunityQueryPersister().persist(it).toV2()
                    .andThen(Single.just(it))
            }
    }
}