package com.ekoapp.ekosdk.internal.repository.post

import androidx.paging.ExperimentalPagingApi
import com.amity.socialcloud.sdk.social.data.post.PostQueryPersister
import com.amity.socialcloud.sdk.social.data.post.PostRemoteDataStore
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.internal.PostEntity
import com.ekoapp.ekosdk.internal.api.dto.EkoPostQueryDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.keycreator.GlobalFeedKeyCreator
import com.ekoapp.ekosdk.internal.mediator.QueryStreamMediator
import com.ekoapp.ekosdk.internal.token.QueryStreamQueryToken
import io.reactivex.Completable
import io.reactivex.Single

@ExperimentalPagingApi
class GlobalFeedMediator : QueryStreamMediator<PostEntity, EkoPostQueryDto>(
    nonce = AmityNonce.GLOBAL_FEED,
    queryStreamKeyCreator = GlobalFeedKeyCreator(),
) {

    override fun forceRefresh(): Boolean = true

    override fun getFirstPageRequest(pageSize: Int): Single<EkoPostQueryDto> {
        return getRequest(limit = pageSize)
    }

    override fun getFetchByTokenRequest(token: String): Single<EkoPostQueryDto> {
        return getRequest(token = token)
    }

    override fun persistResponse(dto: EkoPostQueryDto): Completable {
        return PostQueryPersister().persist(dto).toV2()
    }

    override fun convertResponseToQueryToken(dto: EkoPostQueryDto): QueryStreamQueryToken {
        return QueryStreamQueryToken(
            queryStreamKeyCreator = GlobalFeedKeyCreator(),
            next = dto.paging.next,
            previous = dto.paging?.previous,
            primaryKeys = dto.posts?.map { post -> post.postId } ?: listOf()
        )
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<EkoPostQueryDto> {
        return PostRemoteDataStore().getGlobalFeedPosts(
            limit = limit,
            token = token
        ).toV2()
    }
}