package com.ekoapp.ekosdk.internal.repository.stream

import com.ekoapp.ekosdk.EkoObjectRepository
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.stream.StreamSessionSyncState
import com.ekoapp.ekosdk.stream.entity.EkoStreamSessionEntity
import io.reactivex.rxjava3.core.Flowable
import org.joda.time.DateTime

internal class StreamSessionRepository : EkoObjectRepository() {

    fun getStreamSessionCollection(): Flowable<List<EkoStreamSessionEntity>> {
        val streamDao = UserDatabase.get().streamSessionDao()
        return streamDao.getAll(arrayOf(StreamSessionSyncState.PENDING.key,
                StreamSessionSyncState.SYNCING.key,
                StreamSessionSyncState.SYNCED.key))
    }

    fun getStreamSessionCollection(syncStateArray: Array<StreamSessionSyncState>): Flowable<List<EkoStreamSessionEntity>> {
        val streamDao = UserDatabase.get().streamSessionDao()
        val syncStateList = arrayListOf<String>()
        syncStateArray.forEach { syncStateList.add(it.key) }
        return streamDao.getAll(syncStateList.toTypedArray())
    }

    fun getById(sessionId: String): Flowable<EkoStreamSessionEntity> {
        val streamDao = UserDatabase.get().streamSessionDao()
        return streamDao.getById(sessionId)
    }

    fun resetSyncState() {
        val streamDao = UserDatabase.get().streamSessionDao()
        streamDao.updateSyncState(StreamSessionSyncState.SYNCING, StreamSessionSyncState.PENDING)
    }

    fun createStreamSession(sessionId: String, streamId: String,
                            title: String, startedAt: DateTime, resolution: String) {
        val streamDao = UserDatabase.get().streamSessionDao()
        val streamSession = EkoStreamSessionEntity()
        streamSession.sessionId = sessionId
        streamSession.streamId = streamId
        streamSession.title = title
        streamSession.startTime = startedAt
        streamSession.resolution = resolution
        streamDao.insert(streamSession)
    }

    fun updateStreamSession(streamEntity: EkoStreamSessionEntity) {
        val streamDao = UserDatabase.get().streamSessionDao()
        streamDao.update(streamEntity)
    }

    fun markAsSyncing(streamEntities: List<EkoStreamSessionEntity>) {
        val streamDao = UserDatabase.get().streamSessionDao()
        streamEntities.forEach { it.syncState = StreamSessionSyncState.SYNCING.key }
        streamDao.update(streamEntities)
    }

    fun markAsSynced(streamEntities: List<EkoStreamSessionEntity>) {
        val streamDao = UserDatabase.get().streamSessionDao()
        streamEntities.forEach { it.syncState = StreamSessionSyncState.SYNCED.key }
        streamDao.update(streamEntities)
    }

    fun markAsFailed(streamEntities: List<EkoStreamSessionEntity>) {
        val streamDao = UserDatabase.get().streamSessionDao()
        streamEntities.forEach {
            it.syncState = StreamSessionSyncState.PENDING.key
            it.retryCount++
        }
        streamDao.update(streamEntities)
        deleteIfExceedRetry(streamEntities)
    }

    private fun deleteIfExceedRetry(streamEntities: List<EkoStreamSessionEntity>) {
        val streamDao = UserDatabase.get().streamSessionDao()
        streamEntities.forEach {
            if (it.retryCount > MAX_RETRY) {
                streamDao.delete(it)
            }
        }
    }
}

const val MAX_RETRY = 3