package com.ekoapp.ekosdk.internal.repository.user

import androidx.paging.ExperimentalPagingApi
import com.amity.socialcloud.sdk.AmityCoreClient
import com.amity.socialcloud.sdk.core.data.follow.FollowQueryPersister
import com.amity.socialcloud.sdk.core.data.follow.FollowRemoteDataStore
import com.amity.socialcloud.sdk.core.user.AmityFollowRelationship
import com.amity.socialcloud.sdk.core.user.AmityFollowStatusFilter
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.internal.api.dto.EkoFollowQueryDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.data.model.EkoUserFollowStatusEntity
import com.ekoapp.ekosdk.internal.keycreator.FollowerKeyCreator
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import com.ekoapp.ekosdk.internal.token.QueryStreamQueryToken
import com.ekoapp.ekosdk.internal.usecase.user.FollowerReactorUseCase
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single

@OptIn(ExperimentalPagingApi::class)
class FollowerMediator(
    val userId: String,
    val filter: AmityFollowStatusFilter
) :
    DynamicQueryStreamMediator<EkoUserFollowStatusEntity, EkoFollowQueryDto, AmityFollowRelationship>(
        dynamicQueryStreamKeyCreator = FollowerKeyCreator(
            userId = userId,
            followStatusFilter = filter
        ),
        nonce = AmityNonce.FOLLOWER_LIST
    ) {

    override fun provideReactorPublisher(): Flowable<AmityFollowRelationship> {
        return FollowerReactorUseCase().execute(
            userId = userId,
            filter = filter
        ).toV2()
    }

    override fun getFirstPageRequest(pageSize: Int): Single<EkoFollowQueryDto> {
        return getRequest(limit = pageSize)
    }

    override fun getFetchByTokenRequest(token: String): Single<EkoFollowQueryDto> {
        return getRequest(token = token)
    }

    override fun persistResponse(dto: EkoFollowQueryDto): Completable {
        return FollowQueryPersister().persist(dto).toV2()
    }

    override fun convertResponseToQueryToken(dto: EkoFollowQueryDto): QueryStreamQueryToken {
        return DynamicQueryStreamQueryToken(
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            next = dto.paging?.next,
            previous = dto.paging?.previous,
            primaryKeys = dto.follows.map { it.getId() }
        )
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<EkoFollowQueryDto> {
        return if (userId == AmityCoreClient.getUserId()) {
            FollowRemoteDataStore().getMyFollowers(
                filter = filter,
                limit = limit,
                token = token
            ).toV2()
        } else {
            FollowRemoteDataStore().getUserFollowers(
                userId = userId,
                filter = filter,
                limit = limit,
                token = token
            ).toV2()
        }
    }
}

