package com.ekoapp.ekosdk.internal.repository.user.follow

import androidx.paging.ExperimentalPagingApi
import co.amity.rxremotemediator.AmityQueryTokenDao
import co.amity.rxremotemediator.PageKeyedRxRemoteMediator
import com.amity.socialcloud.sdk.AmityCoreClient
import com.amity.socialcloud.sdk.core.user.AmityFollowStatusFilter
import com.ekoapp.ekosdk.internal.api.EkoSocket
import com.ekoapp.ekosdk.internal.api.dto.EkoFollowQueryDto
import com.ekoapp.ekosdk.internal.api.socket.call.Call
import com.ekoapp.ekosdk.internal.api.socket.call.FollowerPagingQueryConverter
import com.ekoapp.ekosdk.internal.api.socket.request.FollowerQueryRequest
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.data.model.EkoFollowerQueryToken
import com.ekoapp.ekosdk.internal.data.model.EkoUserFollowStatusEntity
import io.reactivex.Single

@ExperimentalPagingApi
internal class FollowerRxRemoteMediator(
    private val userId: String,
    private val filter: AmityFollowStatusFilter,
    tokenDao: AmityQueryTokenDao
) : PageKeyedRxRemoteMediator<EkoUserFollowStatusEntity, EkoFollowerQueryToken>(
    nonce = AmityNonce.USER_FOLLOW,
    queryParameters = mapOf(
        "userId" to userId,
        "filter" to filter
    ),
    tokenDao = tokenDao
) {

    override fun fetchFirstPage(pageSize: Int): Single<EkoFollowerQueryToken> {
        val options = FollowerQueryRequest.FollowQueryOptions(limit = pageSize)
        val request = if (userId == AmityCoreClient.getUserId()) {
            FollowerQueryRequest(userId = null, status = filter.apiKey, options = options)
        } else {
            FollowerQueryRequest(userId = userId, status = filter.apiKey, options = options)
        }
        return call(request)
            .map {
                EkoFollowerQueryToken(
                    userId = userId,
                    filter = filter,
                    next = it.paging?.next,
                    previous = it.paging?.previous,
                    ids = it.follows?.map { follow ->
                        follow.getId()
                    } ?: listOf()
                )
            }
    }

    override fun fetchByToken(token: String): Single<EkoFollowerQueryToken> {
        return fetchDataByToken(token)
            .map {
                EkoFollowerQueryToken(
                    userId = userId,
                    filter = filter,
                    next = it.paging?.next,
                    previous = it.paging?.previous,
                    ids = it.follows?.map { follow ->
                        follow.getId()
                    } ?: listOf()
                )
            }
    }

    private fun fetchDataByToken(token: String): Single<EkoFollowQueryDto> {
        val options = FollowerQueryRequest.FollowQueryOptions(token = token)
        val request = if (userId == AmityCoreClient.getUserId()) {
            FollowerQueryRequest(userId = null, status = filter.apiKey, options = options)
        } else {
            FollowerQueryRequest(userId = userId, status = filter.apiKey, options = options)
        }
        return call(request)
    }

    private fun call(request: FollowerQueryRequest): Single<EkoFollowQueryDto> {
        return EkoSocket.call(Call.create(request, FollowerPagingQueryConverter()))
    }
}