package com.ekoapp.ekosdk.internal.repository.user.follow

import androidx.paging.ExperimentalPagingApi
import co.amity.rxremotemediator.AmityQueryTokenDao
import co.amity.rxremotemediator.PageKeyedRxRemoteMediator
import com.amity.socialcloud.sdk.AmityCoreClient
import com.amity.socialcloud.sdk.core.user.AmityFollowStatusFilter
import com.ekoapp.ekosdk.internal.api.EkoSocket
import com.ekoapp.ekosdk.internal.api.dto.EkoFollowQueryDto
import com.ekoapp.ekosdk.internal.api.socket.call.Call
import com.ekoapp.ekosdk.internal.api.socket.call.FollowingPagingQueryConverter
import com.ekoapp.ekosdk.internal.api.socket.request.FollowingQueryRequest
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.data.model.EkoFollowingQueryToken
import com.ekoapp.ekosdk.internal.data.model.EkoUserFollowStatusEntity
import io.reactivex.Single

@ExperimentalPagingApi
internal class FollowingRxRemoteMediator(
    private val userId: String,
    private val filter: AmityFollowStatusFilter,
    tokenDao: AmityQueryTokenDao
) : PageKeyedRxRemoteMediator<EkoUserFollowStatusEntity, EkoFollowingQueryToken>(
    nonce = AmityNonce.USER_FOLLOW,
    queryParameters = mapOf(
        "userId" to userId,
        "filter" to filter
    ),
    tokenDao = tokenDao
) {

    override fun fetchFirstPage(pageSize: Int): Single<EkoFollowingQueryToken> {
        val options = FollowingQueryRequest.FollowQueryOptions(limit = pageSize)
        val request = if (userId == AmityCoreClient.getUserId()) {
            FollowingQueryRequest(userId = null, status = filter.apiKey, options = options)
        } else {
            FollowingQueryRequest(userId = userId, status = filter.apiKey, options = options)
        }
        return call(request)
            .flatMap {
                Single.just(
                    EkoFollowingQueryToken(
                        userId = userId,
                        filter = filter,
                        next = it.paging?.next,
                        previous = it.paging?.previous,
                        ids = it.follows?.map { follow ->
                            follow.getId()
                        } ?: listOf()
                    )
                )
            }
    }

    override fun fetchByToken(token: String): Single<EkoFollowingQueryToken> {
        return fetchDataByToken(token)
            .flatMap {
                Single.just(
                    EkoFollowingQueryToken(
                        userId = userId,
                        filter = filter,
                        next = it.paging?.next,
                        previous = it.paging?.previous,
                        ids = it.follows?.map { follow ->
                            follow.getId()
                        } ?: listOf()
                    )
                )
            }
    }

    private fun fetchDataByToken(token: String): Single<EkoFollowQueryDto> {
        val options = FollowingQueryRequest.FollowQueryOptions(token = token)
        val request = if (userId == AmityCoreClient.getUserId()) {
            FollowingQueryRequest(userId = null, status = filter.apiKey, options = options)
        } else {
            FollowingQueryRequest(userId = userId, status = filter.apiKey, options = options)
        }
        return call(request)
    }

    private fun call(request: FollowingQueryRequest): Single<EkoFollowQueryDto> {
        return EkoSocket.call(Call.create(request, FollowingPagingQueryConverter()))
    }
}