package com.ekoapp.ekosdk.internal.usecase.stream

import com.ekoapp.core.utils.toV3
import com.ekoapp.ekosdk.AmityNetworkUtils
import com.ekoapp.ekosdk.internal.api.EkoApi
import com.ekoapp.ekosdk.internal.api.EkoStreamApi
import com.ekoapp.ekosdk.internal.api.http.request.StreamSessionRequestCreator
import com.ekoapp.ekosdk.internal.data.EkoDatabase
import com.ekoapp.ekosdk.internal.data.model.EkoApiKey
import com.ekoapp.ekosdk.internal.repository.stream.StreamSessionRepository
import com.ekoapp.ekosdk.stream.entity.EkoStreamSessionEntity
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Maybe
import java.util.concurrent.TimeUnit

class SendStreamSessionUseCase {

    private var retryCount = 0

    fun execute(streamSessions: List<EkoStreamSessionEntity>): Completable {
        return getApiKey()
                .flatMapCompletable { sendToServer(it.apiKey, streamSessions) }
                .doOnComplete { markAsSynced(streamSessions) }
                .doOnError { markAsFailed(streamSessions) }
    }

    fun execute(streamSessions: List<EkoStreamSessionEntity>, requireNetwork: Boolean): Completable {
        return when {
            streamSessions.isEmpty() -> {
                Completable.complete()
            }
            requireNetwork -> {
                Flowable.interval(1, TimeUnit.SECONDS)
                        .filter { shouldSend() }
                        .takeUntil { retryCount >= ONE_MINUTE }
                        .firstOrError()
                        .flatMapCompletable { execute(streamSessions) }
            }
            else -> {
                execute(streamSessions)
            }
        }
    }

    private fun shouldSend(): Boolean {
        return if (AmityNetworkUtils.isNetworkAvailable()) {
            true
        } else {
            retryCount++
            false
        }
    }

    private fun getApiKey(): Maybe<EkoApiKey> {
        return EkoDatabase.get().apiKeyDao().currentApiKey.toV3()
    }

    private fun sendToServer(apiKey: String, streamSessions: List<EkoStreamSessionEntity>): Completable {
        StreamSessionRepository().markAsSyncing(streamSessions)
        val request = StreamSessionRequestCreator(streamSessions).create()
        return EkoApi.get(EkoStreamApi::class)
            .flatMapCompletable {
                it.sendStreamSession(apiKey, request).toV3()
            }

    }

    private fun markAsSynced(streamSessions: List<EkoStreamSessionEntity>) {
        StreamSessionRepository().markAsSynced(streamSessions)
    }

    private fun markAsFailed(streamSessions: List<EkoStreamSessionEntity>) {
        StreamSessionRepository().markAsFailed(streamSessions)
    }
}

private const val ONE_MINUTE = 60