package com.amity.socialcloud.sdk.chat.message

import android.os.Parcelable
import com.amity.socialcloud.sdk.core.AmityFile
import com.amity.socialcloud.sdk.core.AmityTags
import com.amity.socialcloud.sdk.core.AmityVideo
import com.amity.socialcloud.sdk.core.JsonObjectParceler
import com.amity.socialcloud.sdk.core.data.file.FileRepository
import com.amity.socialcloud.sdk.core.file.AmityAudio
import com.amity.socialcloud.sdk.core.file.AmityFileInfo
import com.amity.socialcloud.sdk.core.file.AmityImage
import com.amity.socialcloud.sdk.core.mention.AmityMentionee
import com.amity.socialcloud.sdk.core.reaction.AmityMessageReactionQuery
import com.amity.socialcloud.sdk.core.reaction.AmityReactor
import com.amity.socialcloud.sdk.core.reaction.ReactionReferenceType
import com.amity.socialcloud.sdk.core.user.AmityUser
import com.amity.socialcloud.sdk.socket.util.EkoGson
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.ReactorObject
import com.ekoapp.ekosdk.internal.data.model.AmityReactionMap
import com.ekoapp.ekosdk.internal.usecase.message.MessageDeleteUseCase
import com.ekoapp.ekosdk.message.flag.AmityMessageFlagger
import com.google.common.base.Objects
import com.google.gson.JsonObject
import io.reactivex.Completable
import kotlinx.android.parcel.Parcelize
import kotlinx.android.parcel.TypeParceler
import org.amity.types.ObjectId
import org.joda.time.DateTime

@Parcelize
@TypeParceler<JsonObject?, JsonObjectParceler>
data class AmityMessage internal constructor(
    private val messageId: String = ObjectId.get().toHexString(),
    private val channelId: String = "",
    private val userId: String = "",
    private val parentId: String?,
    private val channelSegment: Int = 0,
    private val childrenNumber: Int = 0,
    private val editedAt: DateTime? = null,
    private val isDeleted: Boolean = false,
    private val readByCount: Int = 0,
    private val flagCount: Int = 0,
    private val tags: AmityTags = AmityTags(),
    internal var myReactions: List<String> = emptyList(),
    private val reactions: AmityReactionMap = AmityReactionMap(),
    private val reactionCount: Int = 0,
    internal var user: AmityUser? = null,
    private val type: DataType = DataType.CUSTOM,
    private val data: JsonObject?,
    private val syncState: String = State.SYNCED.stateName,
    private val createdAt: DateTime,
    private val updatedAt: DateTime,
    private val metadata: JsonObject?,
    internal var mentionees: List<AmityMentionee>,
    internal var isFlaggedByMe: Boolean = false,
    internal val fileId: String? = null,
    internal var file: AmityFileInfo? = null,
    internal var thumbnailFile: AmityImage? = null
): Parcelable, ReactorObject {

    fun getMessageId(): String {
        return messageId
    }

    fun getChannelId(): String {
        return channelId
    }

    fun getUserId(): String {
        return userId
    }

    fun getParentId(): String? {
        return parentId
    }

    fun getChannelSegment(): Int {
        return channelSegment
    }

    fun getChildrenNumber(): Int {
        return childrenNumber
    }

    fun getEditedAt(): DateTime {
        return editedAt ?: DateTime.now()
    }

    fun isDeleted(): Boolean {
        return isDeleted
    }

    fun getReadByCount(): Int {
        return readByCount
    }

    fun getFlagCount(): Int {
        return flagCount
    }

    fun getTags(): AmityTags {
        return tags
    }

    fun getMyReactions(): List<String> {
        return myReactions
    }

    fun getReactionMap(): AmityReactionMap {
        return reactions
    }

    fun getReactionCount(): Int {
        return reactionCount
    }

    fun getDataType(): DataType {
        return type
    }

    fun getCreatedAt(): DateTime {
        return createdAt
    }

    fun getUpdatedAt(): DateTime {
        return updatedAt
    }

    fun getUser(): AmityUser? {
        return user
    }

    fun getData(): Data {
        return Data.from(messageId, getDataType(), data ?: JsonObject(), file, thumbnailFile)
    }

    fun getState(): State {
        return State.enumOf(syncState)
    }

    fun isEdited(): Boolean {
        return editedAt?.isAfter(createdAt) ?: false
    }

    fun delete(): Completable {
        return MessageDeleteUseCase().execute(messageId).toV2()
    }

    fun isFlaggedByMe(): Boolean {
        return isFlaggedByMe
    }

    fun getMetadata(): JsonObject? {
        return metadata
    }

    fun getMentionees(): List<AmityMentionee> {
        return mentionees
    }
    
    fun getThumbnailFileId(): String? {
        return data?.get("thumbnailFileId")?.asString
    }

    fun report(): AmityMessageFlagger {
        return AmityMessageFlagger(messageId)
    }

    fun react(): AmityReactor {
        return AmityReactor(ReactionReferenceType.MESSAGE, messageId)
    }

    fun getReactions(): AmityMessageReactionQuery.Builder {
        return AmityMessageReactionQuery.Builder(messageId)
    }

    override fun updatedAt(): DateTime? {
        return updatedAt
    }

    override fun uniqueId(): String {
        return messageId
    }

    sealed class Data : Parcelable {

        @Parcelize
        class TEXT(
            private val messageId: String,
            private val text: String? = ""
        ) : Data() {

            fun getMessageId(): String {
                return messageId
            }

            fun getText(): String {
                return text ?: ""
            }

            fun edit(): AmityTextMessageEditOption {
                return AmityTextMessageEditOption(getMessageId())
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is TEXT
                        && Objects.equal(other.messageId, messageId)
                        && Objects.equal(other.text, text))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(messageId, text)
            }
        }

        @Parcelize
        class IMAGE(
            private val messageId: String,
            private val caption: String?,
            private val file: AmityFileInfo? = null
        ) : Data() {

            fun getMessageId(): String {
                return messageId
            }

            fun getCaption(): String? {
                return caption
            }

            fun getImage(): AmityImage? {
                return file?.let {
                    it as AmityImage
                }
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is IMAGE
                        && Objects.equal(other.messageId, messageId)
                        && Objects.equal(other.caption, caption)
                        && Objects.equal(other.file, file))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(messageId, caption, file)
            }
        }

        @Parcelize
        class FILE(
            private val messageId: String,
            private val caption: String?,
            private val file: AmityFileInfo? = null
        ) : Data() {

            fun getMessageId(): String {
                return messageId
            }

            fun getCaption(): String? {
                return caption
            }

            fun getFile(): AmityFile? {
                return file?.let {
                    it as AmityFile
                }
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is FILE
                        && Objects.equal(other.messageId, messageId)
                        && Objects.equal(other.caption, caption)
                        && Objects.equal(other.file, file))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(messageId, caption, file)
            }

        }

        @Parcelize
        class AUDIO(
            private val messageId: String,
            private val file: AmityFileInfo? = null
        ) : Data() {

            fun getMessageId(): String {
                return messageId
            }

            fun getAudio(): AmityAudio? {
                return file?.let {
                    it as AmityAudio
                }
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is AUDIO
                        && Objects.equal(other.messageId, messageId)
                        && Objects.equal(other.file, file))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(messageId, file)
            }

        }
    
        @Parcelize
        class VIDEO(
            private val messageId: String,
            private val file: AmityFileInfo? = null,
            private val thumbnailImageFile: AmityImage? = null
        ) : Data() {
        
            fun getMessageId(): String {
                return messageId
            }
        
            fun getVideo(): AmityVideo? {
                return file?.let {
                    it as AmityVideo
                }
            }
            
            fun getThumbnailImage(): AmityImage? {
                return thumbnailImageFile
            }
        
            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is VIDEO
                        && Objects.equal(other.messageId, messageId)
                        && Objects.equal(other.file, file))
            }
        
            override fun hashCode(): Int {
                return Objects.hashCode(messageId, file)
            }
        
        }

        @Parcelize
        @TypeParceler<JsonObject, JsonObjectParceler>
        class CUSTOM(private val data: JsonObject) : Data() {

            fun getMessageId(): String {
                return data.get("messageId").asString
            }

            fun getRawData(): JsonObject {
                return data
            }

            fun <T> getSerializedData(clazz: Class<T>): T? {
                return EkoGson.get().fromJson(data, clazz)
            }

            fun edit(): AmityCustomTextMessageEditOption {
                return AmityCustomTextMessageEditOption(getMessageId())
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is CUSTOM
                        && Objects.equal(other.data, data))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(data)
            }

        }

        companion object {
            internal fun from(
                    messageId: String,
                    dataType: DataType,
                    data: JsonObject,
                    file: AmityFileInfo?,
                    thumbnailFile: AmityImage?
            ): Data {
                val caption = data.get("caption")?.asString ?: ""
                return when (dataType) {
                    DataType.TEXT -> {
                        TEXT(messageId, data.get("text")?.asString ?: "")
                    }
                    DataType.IMAGE -> {
                        IMAGE(messageId, caption, file)
                    }
                    DataType.FILE -> {
                        FILE(messageId, caption, file)
                    }
                    DataType.AUDIO -> {
                        AUDIO(messageId, file)
                    }
                    DataType.VIDEO -> {
                        VIDEO(messageId, file, thumbnailFile)
                    }
                    else -> {
                        CUSTOM(data)
                    }
                }
            }
        }
    }

    enum class State(val stateName: String) {
        CREATED("created"),
        UPLOADING("uploading"),
        SYNCING("syncing"),
        SYNCED("synced"),
        FAILED("failed");

        companion object {
            fun enumOf(value: String): State = values().find { it.stateName == value } ?: SYNCED
        }
    }

    enum class DataType(val apiKey: String) {
        TEXT("text"),
        IMAGE("image"),
        FILE("file"),
        AUDIO("audio"),
        VIDEO("video"),
        CUSTOM("custom");

        companion object {
            fun enumOf(value: String): DataType {
                return values().find { it.apiKey == value } ?: CUSTOM
            }
        }
    }

}