package com.amity.socialcloud.sdk.chat.message

import com.amity.socialcloud.sdk.core.AmityTags
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.internal.usecase.message.MessageManualFirstPageLoadUseCase
import com.ekoapp.ekosdk.internal.usecase.message.MessageManualLoadWithTokenUsecase
import com.ekoapp.ekosdk.internal.usecase.message.MessageManualQueryUseCase
import io.reactivex.BackpressureStrategy
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.schedulers.Schedulers
import io.reactivex.subjects.BehaviorSubject

private const val MESSAGE_LOADER_PAGE_SIZE = 20

class AmityMessageLoader internal constructor(
    private val channelId: String,
    private val stackFromEnd: Boolean,
    private val parentId: String?,
    private val isFilterByParentId: Boolean,
    private var isDeleted: Boolean?,
    private val includingTags: AmityTags,
    private val excludingTags: AmityTags,
    private val type: AmityMessage.DataType?
) {

    private var ids = mutableListOf<String>()
    private val idPublisher = BehaviorSubject.create<List<String>>()

    private var token: String? = null
    private var isLoading = false

    fun hasMore(): Boolean {
        return token?.isNotEmpty() ?: true
    }

    fun getResult(): Flowable<List<AmityMessage>> {
        return idPublisher
            .toFlowable(BackpressureStrategy.BUFFER)
            .switchMap {
                MessageManualQueryUseCase().execute(
                    channelId,
                    parentId,
                    isFilterByParentId,
                    isDeleted,
                    includingTags,
                    excludingTags,
                    type,
                    ids,
                    hasMore()
                ).toV2()
            }.map { list ->
                ids = list.map { it.getMessageId() }.toMutableList()
                return@map list
            }
            .doOnSubscribe {
                idPublisher.onNext(ids)
            }
    }

    fun load(): Completable {
        return makeRequest()
    }

    private fun makeRequest(): Completable {
        if (isLoading) return Completable.never()
        isLoading = true
        return when {
            token == null -> {
                MessageManualFirstPageLoadUseCase().execute(
                    channelId = channelId,
                    stackFromEnd = stackFromEnd,
                    parentId = parentId,
                    isFilterByParentId = isFilterByParentId,
                    isDeleted = isDeleted,
                    includingTags = includingTags,
                    excludingTags = excludingTags,
                    type = type,
                    limit = getPageSize()
                ).toV2()
                    .subscribeOn(Schedulers.io())
                    .doOnSuccess {
                        token = it.token
                        ids.addAll(it.ids)
                        idPublisher.onNext(ids)
                    }.ignoreElement()
            }
            token!!.isNotEmpty() -> {
                MessageManualLoadWithTokenUsecase().execute(
                    channelId = channelId,
                    stackFromEnd = stackFromEnd,
                    parentId = parentId,
                    isFilterByParentId = isFilterByParentId,
                    isDeleted = isDeleted,
                    includingTags = includingTags,
                    excludingTags = excludingTags,
                    type = type,
                    token = token!!
                ).toV2()
                    .subscribeOn(Schedulers.io())
                    .doOnSuccess {
                        token = it.token
                        ids.addAll(it.ids)
                        idPublisher.onNext(ids)
                    }.ignoreElement()
            }
            else -> {
                Completable.complete()
            }
        }.doOnTerminate {
            isLoading = false
        }.doOnDispose {
            isLoading = false
        }
    }

    private fun getPageSize(): Int {
        return MESSAGE_LOADER_PAGE_SIZE
    }
}