package com.amity.socialcloud.sdk.chat.message


import androidx.paging.PagedList
import androidx.paging.PagingData
import com.amity.socialcloud.sdk.core.AmityTags
import com.amity.socialcloud.sdk.core.ExperimentalAmityApi
import com.ekoapp.core.utils.toV2
import com.ekoapp.ekosdk.internal.usecase.message.GetMessagePagedListUseCase
import com.ekoapp.ekosdk.internal.usecase.message.GetMessagePagingDataUseCase
import io.reactivex.Flowable

class AmityMessageQuery internal constructor(
    private val channelId: String,
    private val stackFromEnd: Boolean,
    private val parentId: String?,
    private val isFilterByParentId: Boolean,
    private var isDeleted: Boolean?,
    private val includingTags: AmityTags,
    private val excludingTags: AmityTags,
    private val type: AmityMessage.DataType?
) {

    class Builder {
        private lateinit var channelId: String
        private var stackFromEnd: Boolean = true
        private var parentId: String? = null
        private var isFilterByParentId: Boolean = false
        private var includingTags: AmityTags = AmityTags()
        private var excludingTags: AmityTags = AmityTags()
        private var isDeleted: Boolean? = null
        private var type: AmityMessage.DataType? = null

        @JvmSynthetic
        internal fun channelId(channelId: String): Builder {
            this.channelId = channelId
            return this
        }

        fun includeDeleted(includeDeleted: Boolean): Builder {
            if (!includeDeleted) {
                isDeleted = false
            }
            return this
        }

        fun stackFromEnd(stackFromEnd: Boolean): Builder {
            return apply { this.stackFromEnd = stackFromEnd }
        }

        fun parentId(parentId: String?): Builder {
            return apply {
                this.parentId = parentId
                this.isFilterByParentId = true
            }
        }

        fun includingTags(includingTags: AmityTags): Builder {
            return apply { this.includingTags = includingTags }
        }

        fun excludingTags(excludingTags: AmityTags): Builder {
            return apply { this.excludingTags = excludingTags }
        }
    
        fun type(type: AmityMessage.DataType): Builder {
            return apply { this.type = type }
        }

        fun build(): AmityMessageQuery {
            return AmityMessageQuery(
                channelId,
                stackFromEnd,
                parentId,
                isFilterByParentId,
                isDeleted,
                includingTags,
                excludingTags,
                type
            )
        }
    }

    fun query(): Flowable<PagedList<AmityMessage>> {
        return GetMessagePagedListUseCase().execute(
            channelId,
            stackFromEnd,
            parentId,
            isFilterByParentId,
            isDeleted,
            includingTags,
            excludingTags,
            type
        ).toV2()
    }

    @ExperimentalAmityApi
    fun loader(): AmityMessageLoader {
        return AmityMessageLoader(
            channelId,
            stackFromEnd,
            parentId,
            isFilterByParentId,
            isDeleted,
            includingTags,
            excludingTags,
            type
        )
    }

    fun getPagingData(): Flowable<PagingData<AmityMessage>> {
        return GetMessagePagingDataUseCase().execute(
            channelId,
            parentId,
            includingTags,
            excludingTags,
            isFilterByParentId,
            isDeleted,
            true,
            type
        ).toV2()
    }
}