package com.amity.socialcloud.sdk.common


import com.amity.socialcloud.sdk.core.exception.EntityExpiredException
import com.amity.socialcloud.sdk.core.exception.EntityNotFoundException
import com.amity.socialcloud.sdk.log.AmityLog
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.ekoapp.ekosdk.EkoObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers

internal abstract class AmityObjectRepository<Entity: EkoObject, PublicModel: Any> {

    companion object {
        const val DEFAULT_PAGE_SIZE = 20
    }

    abstract fun fetchAndSave(objectId: String): Completable

    abstract fun queryFromCache(objectId: String): Entity?

    abstract fun mapper(): ModelMapper<Entity, PublicModel>

    abstract fun observeFromCache(objectId: String): io.reactivex.rxjava3.core.Flowable<Entity>

    open fun observe(objectId: String): io.reactivex.rxjava3.core.Flowable<PublicModel>{
        return observeFromCache(objectId)
            .map { mapper().map(it) }
    }

    open fun obtain(objectId: String): Single<PublicModel> {
        val TAG = "OTP2"
        val queryFromCacheSingle = Single.fromCallable {
            //TODO Remove all logs once tested
            AmityLog.tag(TAG).e("query cached data")
            return@fromCallable queryFromCache(objectId) ?: throw EntityNotFoundException
        }
        return queryFromCacheSingle
            .subscribeOn(Schedulers.io())
            .flatMap {
                val isNotExpired = it.expiresAt?.isAfterNow == true
                if (isNotExpired) {
                    AmityLog.tag(TAG).e("return cached data")
                    return@flatMap Single.just(it)
                } else {
                    return@flatMap Single.error(EntityExpiredException)
                }
            }
            .onErrorResumeNext {
                AmityLog.tag(TAG).e("${it.message}")
                if (it == EntityNotFoundException || it == EntityExpiredException) {
                    AmityLog.tag(TAG).e("fetching object from server")
                    return@onErrorResumeNext fetchAndSave(objectId)
                            .andThen(Single.defer { queryFromCacheSingle })
                            .onErrorResumeNext { error ->
                                Single.error(AmityException.create("unknown error", error, AmityError.UNKNOWN))
                            }
                } else {
                    return@onErrorResumeNext Single.error(it)
                }
            }.map { mapper().map(it) }
    }

}