package com.amity.socialcloud.sdk.social.domain.story.create

import android.net.Uri
import com.amity.socialcloud.sdk.core.data.file.FileRepository
import com.amity.socialcloud.sdk.model.core.file.AmityImage
import com.amity.socialcloud.sdk.model.core.file.upload.AmityUploadResult
import com.amity.socialcloud.sdk.model.social.story.AmityStory
import com.amity.socialcloud.sdk.model.social.story.AmityStoryImageDisplayMode
import com.amity.socialcloud.sdk.model.social.story.AmityStoryItem
import com.amity.socialcloud.sdk.social.data.story.StoryRepository
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single

class CreateImageStoryUseCase : CreateStoryUseCase {

    fun execute(
        targetType: AmityStory.TargetType,
        targetId: String,
        fileUri: Uri,
        storyItems: List<AmityStoryItem> = emptyList(),
        metadata: JsonObject?,
        imageDisplayMode: AmityStoryImageDisplayMode,
    ): Completable {
        return createStory(
            targetType = targetType,
            targetId = targetId,
            createLocalFile = { uniqueId ->
                createLocalFile(
                    uniqueId = uniqueId,
                    fileUri = fileUri
                )
            },
            createLocalStory = { uniqueId ->
                createLocalStory(
                    uniqueId = uniqueId,
                    targetType = targetType,
                    targetId = targetId,
                    storyItems = storyItems,
                    metadata = metadata,
                    imageDisplayMode = imageDisplayMode
                )
            },
            uploadFile = { uniqueId ->
                uploadFile(
                    uniqueId,
                    fileUri
                )
            },
            createRemoteStory = { uniqueId, fileId ->
                createRemoteStory(
                    uniqueId = uniqueId,
                    targetType = targetType,
                    targetId = targetId,
                    fileId = fileId,
                    storyItems = storyItems,
                    metadata = metadata,
                    imageDisplayMode = imageDisplayMode
                )
            }
        )
    }

    private fun createLocalFile(
        uniqueId: String,
        fileUri: Uri,
    ): Completable {
        return FileRepository().createLocalFile(
            fileId = uniqueId,
            fileType = AmityStory.DataType.IMAGE.apiKey,
            filePath = fileUri.toString()
        )
    }

    private fun createLocalStory(
        uniqueId: String,
        targetType: AmityStory.TargetType,
        targetId: String,
        storyItems: List<AmityStoryItem> = emptyList(),
        metadata: JsonObject?,
        imageDisplayMode: AmityStoryImageDisplayMode
    ): Completable {
        return StoryRepository().createLocalImageStory(
            referenceId = uniqueId,
            targetType = targetType,
            targetId = targetId,
            storyItems = storyItems,
            metadata = metadata,
            imageDisplayMode = imageDisplayMode
        )
    }

    private fun uploadFile(
        uniqueId: String,
        fileUri: Uri
    ): Flowable<AmityUploadResult<AmityImage>> {
        return FileRepository().uploadImage(
            uploadId = uniqueId,
            uri = fileUri
        )
    }

    private fun createRemoteStory(
        uniqueId: String,
        targetType: AmityStory.TargetType,
        targetId: String,
        fileId: String,
        storyItems: List<AmityStoryItem> = emptyList(),
        metadata: JsonObject?,
        imageDisplayMode: AmityStoryImageDisplayMode
    ): Single<String> {
        return StoryRepository().createImageStory(
            referenceId = uniqueId,
            targetType = targetType,
            targetId = targetId,
            fileId = fileId,
            storyItems = storyItems,
            metadata = metadata,
            imageDisplayMode = imageDisplayMode
        )
    }

}