package com.amity.socialcloud.sdk.api.social.community.search

import androidx.paging.PagingData
import com.amity.socialcloud.sdk.api.social.community.query.AmityCommunitySortOption
import com.amity.socialcloud.sdk.model.social.community.AmityCommunity
import com.amity.socialcloud.sdk.model.social.community.AmityCommunityFilter
import com.amity.socialcloud.sdk.social.domain.community.CommunitySearchUseCase
import io.reactivex.rxjava3.core.Flowable

class AmityCommunitySearch private constructor(
    private val keyword: String,
    private val categoryId: String,
    private val filter: AmityCommunityFilter,
    private val sortBy: AmityCommunitySortOption,
    private val isDeleted: Boolean?
) {

    /**
     * Search for communities
     * @return LiveCollection of AmityCommunity as Flowable<PagingData<AmityCommunity>>.
     */
    fun query(): Flowable<PagingData<AmityCommunity>> {
        return CommunitySearchUseCase().execute(
            keyword = keyword,
            categoryId = categoryId,
            filter = filter,
            sortBy = sortBy,
            isDeleted = isDeleted
        )
    }

    class Builder internal constructor(
        private var keyword: String? = null
    ) {
        private var categoryId: String? = null
        private var filter: AmityCommunityFilter = AmityCommunityFilter.ALL
        private var sortBy: AmityCommunitySortOption = AmityCommunitySortOption.DISPLAY_NAME
        private var isDeleted: Boolean? = null

//        /**
//         * @param [keyword] to query by community's displayName
//         * @return the same instance of AmityCommunityQuery.Builder
//         */
//        fun withKeyword(keyword: String): Builder {
//            return apply { this.keyword = keyword }
//        }

        /**
         * @param [categoryId] to filterBy by community's categoryId
         * @return the same instance of AmityCommunityQuery.Builder
         */
        fun categoryId(categoryId: String): Builder {
            return apply { this.categoryId = categoryId }
        }

        /**
         * @param [filter] to specify AmityCommunityFilter
         * when [filter] is AmityCommunityFilter.ALL, the membership filter is not applied
         * when [filter] is AmityCommunityFilter.MEMBER, the result contains only communities that user is a member of
         * when [filter] is AmityCommunityFilter.NOT_MEMBER, the result contains only communities that user is not a member of
         * the default value of [filter] is AmityCommunityFilter.ALL
         * @return the same instance of AmityCommunityQuery.Builder
         */
        fun filter(filter: AmityCommunityFilter): Builder {
            return apply { this.filter = filter }
        }

        /**
         * @param [includeDeleted] to specify includeDeleted filter.
         * When [includeDeleted] is true, the result will include soft-deleted communities.
         * When [includeDeleted] is false, the result will exclude soft-deleted communities.
         * The default value of [includeDeleted] is true.
         * @return the same AmityCommunityQuery.Builder instance.
         */
        fun includeDeleted(includeDeleted: Boolean): Builder {
            return apply {
                this.isDeleted = if (includeDeleted) null else false
            }
        }

        /**
         * @param [sortBy] to specify the sorting option.
         * When [sortBy] is AmityCommunitySortOption.DISPLAY_NAME, posts will be sorted by displayName property ASC.
         * When [sortBy] is AmityCommunitySortOption.FIRST_CREATED, posts will be sorted by createdAt property ASC.
         * When [sortBy] is AmityCommunitySortOption.LAST_CREATED, posts will be sorted by createdAt property DESC.
         * The default value of [sortBy] is AmityCommunitySortOption.DISPLAY_NAME.
         * @return the same AmityCommunityQuery.Builder instance.
         */
        fun sortBy(sortBy: AmityCommunitySortOption): Builder {
            return apply { this.sortBy = sortBy }
        }

        /**
         * Instantiates AmityCommunitySearch with built params.
         * @return new instance of AmityCommunitySearch.
         */
        fun build(): AmityCommunitySearch {
            return AmityCommunitySearch(
                keyword = keyword ?: "",
                categoryId = categoryId ?: "",
                filter = filter,
                sortBy = sortBy,
                isDeleted = isDeleted
            )
        }
    }
}