package com.amity.socialcloud.sdk.core

import android.content.Intent
import android.net.Uri
import com.amity.socialcloud.sdk.chat.data.message.MessageLocalDataStore
import com.amity.socialcloud.sdk.chat.data.message.MessageMentionTargetMapper
import com.amity.socialcloud.sdk.chat.data.message.MessageRepository
import com.amity.socialcloud.sdk.core.data.file.FileRepository
import com.amity.socialcloud.sdk.core.session.component.SessionComponent
import com.amity.socialcloud.sdk.core.session.eventbus.NetworkConnectionEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.core.session.model.NetworkConnectionEvent
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.amity.socialcloud.sdk.model.chat.message.AmityMessage
import com.amity.socialcloud.sdk.model.chat.message.AmityMessageAttachment
import com.amity.socialcloud.sdk.model.core.content.AmityContentFeedType
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.amity.socialcloud.sdk.model.core.file.AmityFileInfo
import com.amity.socialcloud.sdk.model.core.file.upload.AmityUploadResult
import com.ekoapp.ekosdk.internal.EkoMessageEntity
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import com.ekoapp.ekosdk.internal.util.AppContext
import com.google.gson.JsonObject
import io.reactivex.rxjava3.android.schedulers.AndroidSchedulers
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.CompletableEmitter
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.processors.PublishProcessor
import io.reactivex.rxjava3.schedulers.Schedulers
import java.lang.Thread.sleep
import java.util.LinkedList
import java.util.Queue
import java.util.concurrent.Executors
import java.util.concurrent.TimeUnit

internal class MessageSyncEngine(
	sessionLifeCycleEventBus: SessionLifeCycleEventBus,
	sessionStateEventBus: SessionStateEventBus
) : SessionComponent(sessionLifeCycleEventBus, sessionStateEventBus) {
	
	private var isActive = sessionStateEventBus.getCurrentEvent() == SessionState.Established
	private var isOnline = true
	private val messageSyncTrigger: PublishProcessor<Unit> = PublishProcessor.create()
	private val attachmentUploadTrigger: PublishProcessor<Unit> = PublishProcessor.create()
	private val textMessageQueue: Queue<MessageSyncJob> = LinkedList()
	private val attachmentMessageQueue: Queue<MessageSyncJob> = LinkedList()
	private val singleThreadSchedulers = Schedulers.from(Executors.newSingleThreadExecutor())
	
	// Timer every 5 seconds to ensure that no sync job is missed
	private val insurerSyncTimer = Flowable.interval(SYNC_INSURER_INTERVAL_MS, TimeUnit.MILLISECONDS)
			.skipWhile { textMessageQueue.isEmpty() && attachmentMessageQueue.isEmpty() }
			.flatMapCompletable {
				Completable.fromCallable {
					if (isActive && isOnline && (textMessageQueue.isNotEmpty() || attachmentMessageQueue.isNotEmpty())) {
						syncMessageJob()
						uploadAttachmentJobs()
					}
				}
			}
			.onErrorComplete()
			.subscribeOn(Schedulers.io())
	
	init {
		NetworkConnectionEventBus
			.observe()
			.doOnNext { event  ->
				when (event) {
					is NetworkConnectionEvent.Connected -> {
						isOnline = true
						syncMessageJob()
						uploadAttachmentJobs()
					}
					else -> {
						isOnline = false
					}
				}
			}
				.subscribeOn(Schedulers.io())
				.subscribe()
		
		insurerSyncTimer.subscribe()
		
		messageSyncTrigger
			.flatMapCompletable {
				if (isActive && isOnline) {
					val job = getTextMessageSyncJob()
					if (job != null) {
						when(job.status) {
							MessageSyncJob.Status.CREATED -> {
								if (job.retryCount > MAX_RETRY_SYNC) {
									MessageLocalDataStore().updateMessageState(job.message.messageId, AmityMessage.State.FAILED)
										.andThen(
											Completable.fromCallable {
												textMessageQueue.remove(job)
												syncMessageJob()
											}
										)
								} else {
									syncMessage(job)
								}
							}
							MessageSyncJob.Status.SYNCED, MessageSyncJob.Status.FAILED  -> {
								Completable.fromCallable {
									textMessageQueue.remove(job)
									syncMessageJob()
								}
							}
							else -> {
								Completable.complete()
							}
						}
					} else {
						Completable.complete()
					}
				} else {
					Completable.complete()
				}
			}
			.onErrorComplete()
			.subscribeOn(singleThreadSchedulers)
			.subscribe()
		
		messageSyncTrigger
			.flatMapCompletable {
				if (isActive && isOnline) {
					val job = getAttachmentMessageSyncJob()
					if (job != null) {
						when(job.status) {
							MessageSyncJob.Status.UPLOADED -> {
								if (job.retryCount > MAX_RETRY_SYNC) {
									MessageLocalDataStore().updateMessageState(job.message.messageId, AmityMessage.State.FAILED)
										.andThen(
											Completable.fromCallable {
												attachmentMessageQueue.remove(job)
												syncMessageJob()
											}
										)
								} else {
									syncMessage(job)
								}
							}
							MessageSyncJob.Status.SYNCED, MessageSyncJob.Status.FAILED -> {
								Completable.fromCallable {
									attachmentMessageQueue.remove(job)
									syncMessageJob()
								}
							}
							else -> {
								Completable.complete()
							}
						}
					} else {
						Completable.complete()
					}
				} else {
					Completable.complete()
				}
			}
			.onErrorComplete()
			.subscribeOn(singleThreadSchedulers)
			.subscribe()
		
		attachmentUploadTrigger
			.flatMapCompletable {
				val availableSlot = attachmentMessageQueue
					.filter { it.status == MessageSyncJob.Status.UPLOADING }
					.let {
						MAX_CONCURRENT_UPLOAD - it.size
					}.let { if (it < 0) 0 else it }
					
				if (availableSlot > 0) {
					attachmentMessageQueue
						.filter { it.status == MessageSyncJob.Status.CREATED }
						.let {
							val size = minOf(it.size, availableSlot)
							if (size > 0) {
								it.subList(0, size)
							} else {
								it
							}
						}
						.let {
							Flowable.fromIterable(it)
								.flatMapCompletable { job ->
									handleMessageAttachmentUpload(job)
										.onErrorComplete()
										.subscribeOn(Schedulers.io())
								}
						}
				} else {
					Completable.complete()
				}
			}
				.onErrorComplete()
				.subscribeOn(Schedulers.io())
				.subscribe()
	}
	
	private fun syncMessageJob() {
		messageSyncTrigger.onNext(Unit)
	}
	
	fun addTextMessageJob(message: EkoMessageEntity ,emitter: CompletableEmitter) {
		val job = MessageSyncJob(message = message, emitter = emitter, status = MessageSyncJob.Status.CREATED)
		textMessageQueue.add(job)
		syncMessageJob()
	}
	
	private fun syncMessage(job: MessageSyncJob): Completable {
		return MessageRepository().syncMessage(
			messageId = job.message.messageId,
			subChannelId = job.message.subChannelId,
			parentId = job.message.parentId,
			type = job.message.type,
			data = job.message.data ?: JsonObject(),
			tags = job.message.getTags(),
			fileId = job.attachment?.let { (it as? AmityMessageAttachment.FILE_ID)?.fileId },
			metadata = job.message.metadata,
			mentionees = job.message.mentionees.let { MessageMentionTargetMapper().map(it) },
		)
			.doOnComplete {
				job.status = MessageSyncJob.Status.SYNCED
				job.emitter.onComplete()
				syncMessageJob()
			}
			.doOnSubscribe {
				job.status = MessageSyncJob.Status.SYNCING
			}
			.onErrorResumeNext { error ->
				handleException(error, job)
			}
	}
	
	private fun getTextMessageSyncJob(): MessageSyncJob? {
		return textMessageQueue.peek()
	}
	
	private fun getAttachmentMessageSyncJob(): MessageSyncJob? {
		return attachmentMessageQueue.peek()
	}
	
	private fun uploadAttachmentJobs() {
		attachmentUploadTrigger.onNext(Unit)
	}
	fun addAttachmentMessageJob(message: EkoMessageEntity ,emitter: CompletableEmitter, attachment: AmityMessageAttachment) {
		val status = if(attachment is AmityMessageAttachment.URL) {
			MessageSyncJob.Status.CREATED
		} else {
			MessageSyncJob.Status.UPLOADED
		}
		val job = MessageSyncJob(message = message, emitter = emitter, status = status, attachment = attachment)
		attachmentMessageQueue.add(job)
		uploadAttachmentJobs()
	}
	
	fun handleMessageAttachmentUpload(job: MessageSyncJob): Completable {
		val messageId = job.message.messageId
		val dataType = AmityMessage.DataType.enumOf(job.message.type)
		val attachment = job.attachment
		return if (attachment is AmityMessageAttachment.URL) {
			MessageLocalDataStore().updateMessageState(
				messageId,
				AmityMessage.State.UPLOADING
			)
				.andThen(
					when (dataType) {
						AmityMessage.DataType.IMAGE -> {
							FileRepository().uploadImage(messageId, attachment.uri)
								.flatMapCompletable {
									handleUploadResult(
										job = job,
										uploadResult = it
									)
								}
						}
						AmityMessage.DataType.FILE -> {
							FileRepository().uploadFile(messageId, attachment.uri)
								.flatMapCompletable {
									handleUploadResult(
										job = job,
										uploadResult = it
									)
								}
						}
						AmityMessage.DataType.AUDIO -> {
							FileRepository().uploadAudio(messageId, attachment.uri)
								.flatMapCompletable {
									handleUploadResult(
										job = job,
										uploadResult = it
									)
								}
						}
						AmityMessage.DataType.VIDEO -> {
							FileRepository().uploadVideo(
									messageId,
									attachment.uri,
									AmityContentFeedType.MESSAGE
							)
								.flatMapCompletable {
									handleUploadResult(
										job = job,
										uploadResult = it
									)
								}
						}
						else -> {
							// Not a valid case
							Completable.complete()
						}
					}
				)
					.doOnSubscribe {
						job.status = MessageSyncJob.Status.UPLOADING
					}
		} else {
			syncMessageJob()
			Completable.complete()
		}
	}
	
	private fun <T : AmityFileInfo> handleUploadResult(
		job: MessageSyncJob,
		uploadResult: AmityUploadResult<T>
	): Completable {
		return when (uploadResult) {
			is AmityUploadResult.COMPLETE<T> -> {
				val fileId = uploadResult.getFile().getFileId()
				job.attachment = AmityMessageAttachment.FILE_ID(fileId)
				job.status = MessageSyncJob.Status.UPLOADED
				syncMessageJob()
				Completable.complete()
			}
			is AmityUploadResult.ERROR -> {
				handleException(uploadResult.getError(), job)
			}
			is AmityUploadResult.CANCELLED -> {
				job.status = MessageSyncJob.Status.FAILED
				MessageLocalDataStore().updateMessageState(job.message.messageId, AmityMessage.State.FAILED)
					.andThen(
						Completable.error(
							AmityException.create(
								message = "${job.message.type.capitalize()} upload cancelled",
								null,
								AmityError.UNKNOWN
							)
						)
					)
			}
			else -> {
				Completable.complete()
			}
		}
	}
	
	private fun handleException(throwable: Throwable, job: MessageSyncJob): Completable {
		val error =  AmityException.fromThrowable(throwable)
		return when (error.code) {
			AmityError.CONNECTION_ERROR.code -> {
				job.status = MessageSyncJob.Status.CREATED
				Completable.fromCallable {
					sleep(5000)
					syncMessageJob()
				}.subscribeOn(Schedulers.io())
			}
			AmityError.UNKNOWN.code -> {
				if (error.httpStatusCode == 502 || error.httpStatusCode == 503) {
					if (job.attachment is AmityMessageAttachment.FILE_ID) {
						job.status = MessageSyncJob.Status.UPLOADED
					} else {
						job.status = MessageSyncJob.Status.CREATED
					}
					job.retryCount = job.retryCount + 1
					Completable.fromCallable {
						sleep(5000)
						syncMessageJob()
					}.subscribeOn(Schedulers.io())
				} else {
					job.status = MessageSyncJob.Status.FAILED
					job.emitter.onError(error)
					MessageLocalDataStore().updateMessageState(job.message.messageId, AmityMessage.State.FAILED)
						.andThen(Completable.fromCallable {
							syncMessageJob()
						})
				}
			}
			else -> {
				job.status = MessageSyncJob.Status.FAILED
				job.emitter.onError(error)
				MessageLocalDataStore().updateMessageState(job.message.messageId, AmityMessage.State.FAILED)
					.andThen(Completable.fromCallable {
						syncMessageJob()
					})
			}
		}
	}
	
	override fun onSessionStateChange(sessionState: SessionState) {
		when(sessionState) {
			SessionState.Established -> {
				isActive = true
			}
			else -> {
				isActive = false
			}
		}
	}
	
	override fun establish(account: EkoAccount) {
		isActive = true
	}
	
	override fun destroy() {
		isActive = false
	}
	
	override fun handleTokenExpire() {
		isActive = false
	}
	
	class MessageSyncJob(val message: EkoMessageEntity, var attachment: AmityMessageAttachment? = null, val emitter: CompletableEmitter, var status: Status, var retryCount: Int = 0) {
		enum class Status {
			CREATED, UPLOADING, UPLOADED, SYNCING, SYNCED, FAILED
		}
	}
	
	companion object {
		private const val SYNC_INSURER_INTERVAL_MS = 5000L
		private const val MAX_RETRY_SYNC = 3
		private const val MAX_CONCURRENT_UPLOAD = 5
		fun grantPersistableUriPermissionIfNeeded(uri: Uri) {
			if (uri.scheme == "content") {
				try {
					AppContext.get().contentResolver.takePersistableUriPermission(uri, Intent.FLAG_GRANT_READ_URI_PERMISSION)
				} catch (e: Exception) {
					// Fail to grant persistable Uri permission
				}
			}
		}
	}

}