package com.amity.socialcloud.sdk.core.data.follow

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.api.core.AmityCoreClient
import com.amity.socialcloud.sdk.common.AmityObjectRepository
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.core.data.follow.info.MyFollowInfoModelMapper
import com.amity.socialcloud.sdk.core.data.follow.paging.BlockedMediator
import com.amity.socialcloud.sdk.core.data.follow.paging.FollowerMediator
import com.amity.socialcloud.sdk.core.data.follow.paging.FollowingMediator
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.follow.AmityFollowRelationship
import com.amity.socialcloud.sdk.model.core.follow.AmityFollowStatus
import com.amity.socialcloud.sdk.model.core.follow.AmityFollowStatusFilter
import com.amity.socialcloud.sdk.model.core.follow.AmityMyFollowInfo
import com.amity.socialcloud.sdk.model.core.follow.AmityUserFollowInfo
import com.amity.socialcloud.sdk.social.data.follow.FollowRelationshipModelMapper
import com.ekoapp.ekosdk.internal.data.model.EkoFollowCountEntity
import com.ekoapp.ekosdk.internal.data.model.EkoUserFollowStatusEntity
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import com.ekoapp.ekosdk.internal.repository.user.helper.UserRepositoryHelper
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.core.SingleSource

@OptIn(ExperimentalPagingApi::class)
internal class FollowRepository : AmityObjectRepository<EkoFollowCountEntity, AmityMyFollowInfo>() {

    override fun fetchAndSave(objectId: String): Completable {
        return FollowRemoteDataStore().getMyFollowInfo()
            .flatMapCompletable {
                FollowQueryPersister().persist(it)
            }
    }

    override fun queryFromCache(objectId: String): EkoFollowCountEntity? {
        return FollowLocalDataStore().getFollowCount(objectId)
    }

    override fun mapper(): ModelMapper<EkoFollowCountEntity, AmityMyFollowInfo> {
        return MyFollowInfoModelMapper()
    }

    override fun observeFromCache(objectId: String): Flowable<EkoFollowCountEntity> {
        return FollowLocalDataStore().observeFollowCount(objectId)
    }

    fun getUserFollowInfo(userId: String): Flowable<AmityUserFollowInfo> {
        val myUserId = AmityCoreClient.getUserId()

        return FollowRemoteDataStore().getUserFollowInfo(userId)
            .flatMapCompletable {
                FollowQueryPersister().persist(it)
            }.andThen(
                Flowable.combineLatest(
                    FollowLocalDataStore().observeUserFollow(myUserId, userId),
                    FollowLocalDataStore().observeFollowCount(userId)
                ) { followEntity: EkoUserFollowStatusEntity, followCountEntity: EkoFollowCountEntity ->
                    val mapFollow =
                        followEntity.let(UserRepositoryHelper()::attachDataAndMapToExternal)
                    UserRepositoryHelper().attachUserFollowInfoDataAndMapToExternal(
                        mapFollow,
                        followCountEntity
                    )
                }
            )
    }

    fun requestFollow(userId: String): Single<AmityFollowStatus> {
        return FollowRemoteDataStore().requestFollow(userId)
            .flatMap { followListDto ->
                FollowQueryPersister().persist(followListDto)
                    .andThen(Single.create {
                        val status = AmityFollowStatus.enumOf(followListDto.follows.first().status)
                        //  TODO Remove after backend return follow count in payload
                        //  if source user request to follow target user and immediately accepted
                        //  increase source user following count
                        //  increase target user follower count
                        if (status == AmityFollowStatus.ACCEPTED) {
                            val followLocalDataStore = FollowLocalDataStore()
                            val myUserId = AmityCoreClient.getUserId()

                            val sourceFollowCountEntity =
                                followLocalDataStore.getFollowCountByIdNow(myUserId)
                            sourceFollowCountEntity?.followingCount?.let { count ->
                                followLocalDataStore.updateFollowingCount(
                                    userId = myUserId,
                                    followerCount = count.inc()
                                )
                            }

                            val targetFollowCountEntity =
                                followLocalDataStore.getFollowCountByIdNow(userId)
                            targetFollowCountEntity?.followerCount?.let { count ->
                                followLocalDataStore.updateFollowerCount(
                                    userId = userId,
                                    followerCount = count.inc()
                                )
                            }
                        }
                        it.onSuccess(status)
                    })
            }
    }

    fun acceptFollow(userId: String): Completable {
        return FollowRemoteDataStore().acceptFollow(userId)
            .flatMapCompletable {
                FollowQueryPersister().persist(it)
                    .andThen(Completable.fromAction {
                        //  TODO Remove after backend return follow count in payload
                        //  if target user accept source user follow request
                        //  increase source user following count
                        //  increase target user follower count
                        //  decrease target user pending count

                        val followLocalDataStore = FollowLocalDataStore()

                        val sourceFollowCountEntity =
                            followLocalDataStore.getFollowCountByIdNow(userId)
                        sourceFollowCountEntity?.followingCount?.let { count ->
                            followLocalDataStore.updateFollowingCount(
                                userId = userId,
                                followerCount = count.inc()
                            )
                        }

                        val myUserId = AmityCoreClient.getUserId()
                        val targetFollowCountEntity =
                            followLocalDataStore.getFollowCountByIdNow(myUserId)

                        targetFollowCountEntity?.followerCount?.let { count ->
                            followLocalDataStore.updateFollowerCount(
                                userId = myUserId,
                                followerCount = count.inc()
                            )
                        }
                        targetFollowCountEntity?.pendingCount?.let { count ->
                            if (count >= 0) {
                                followLocalDataStore.updatePendingCount(
                                    userId = myUserId,
                                    followerCount = count.dec()
                                )
                            }
                        }
                    })
            }
            .doOnError {
                if (AmityError.from(it) == AmityError.ITEM_NOT_FOUND) {
                    FollowLocalDataStore().updateFollower(userId, AmityFollowStatus.NONE)
                } else {
                    FollowLocalDataStore().updateFollower(userId, AmityFollowStatus.PENDING)
                }
            }
    }

    fun declineFollow(userId: String): Completable {
        return FollowRemoteDataStore().declineFollow(userId)
            .flatMapCompletable {
                FollowQueryPersister().persist(it)
                    .andThen(Completable.fromAction {
                        //  TODO Remove after backend return follow count in payload
                        //  if target user decline source user follow request
                        //  decrease target user pending count
                        val followLocalDataStore = FollowLocalDataStore()
                        val myUserId = AmityCoreClient.getUserId()
                        val targetFollowCountEntity =
                            followLocalDataStore.getFollowCountByIdNow(myUserId)

                        targetFollowCountEntity?.pendingCount?.let { count ->
                            if (count >= 0) {
                                followLocalDataStore.updatePendingCount(
                                    userId = myUserId,
                                    followerCount = count.dec()
                                )
                            }
                        }
                    })
            }
            .doOnError {
                if (AmityError.from(it) == AmityError.ITEM_NOT_FOUND) {
                    FollowLocalDataStore().updateFollower(userId, AmityFollowStatus.NONE)
                } else {
                    FollowLocalDataStore().updateFollower(userId, AmityFollowStatus.PENDING)
                }
            }
    }

    fun removeFollower(userId: String): Completable {
        return FollowRemoteDataStore().declineFollow(userId)
            .flatMapCompletable {
                FollowQueryPersister().persist(it)
                    .andThen(Completable.fromAction {
                        //  TODO Remove after backend return follow count in payload
                        //  if target user remove source user from following
                        //  decrease source user following count
                        //  decrease target user follower count
                        val followLocalDataStore = FollowLocalDataStore()
                        val myUserId = AmityCoreClient.getUserId()

                        val sourceFollowCountEntity =
                            followLocalDataStore.getFollowCountByIdNow(userId)

                        sourceFollowCountEntity?.followingCount?.let { count ->
                            if (count >= 0) {
                                followLocalDataStore.updateFollowingCount(
                                    userId = userId,
                                    followerCount = count.dec()
                                )
                            }
                        }

                        val targetFollowCountEntity =
                            followLocalDataStore.getFollowCountByIdNow(myUserId)

                        targetFollowCountEntity?.followerCount?.let { count ->
                            if (count >= 0) {
                                followLocalDataStore.updateFollowerCount(
                                    userId = myUserId,
                                    followerCount = count.dec()
                                )
                            }
                        }
                    })
            }
            .doOnError {
                if (AmityError.from(it) == AmityError.ITEM_NOT_FOUND) {
                    FollowLocalDataStore().updateFollower(userId, AmityFollowStatus.NONE)
                } else {
                    FollowLocalDataStore().updateFollower(userId, AmityFollowStatus.ACCEPTED)
                }
            }
    }

    fun unfollow(userId: String): Completable {
        return FollowRemoteDataStore().unfollow(userId)
            .flatMapCompletable {
                FollowQueryPersister().persist(it)
                    .andThen(Completable.fromAction {
                        //  TODO Remove after backend return follow count in payload
                        //  if source user unfollow target user
                        //  decrease source user following count
                        //  decrease target user follower count
                        val followLocalDataStore = FollowLocalDataStore()
                        val myUserId = AmityCoreClient.getUserId()

                        val sourceFollowCountEntity =
                            followLocalDataStore.getFollowCountByIdNow(myUserId)
                        sourceFollowCountEntity?.followingCount?.let { count ->
                            if (count >= 0) {
                                followLocalDataStore.updateFollowingCount(
                                    userId = myUserId,
                                    followerCount = count.dec()
                                )
                            }
                        }

                        val targetFollowCountEntity =
                            followLocalDataStore.getFollowCountByIdNow(userId)
                        targetFollowCountEntity?.followerCount?.let { count ->
                            if (count >= 0) {
                                followLocalDataStore.updateFollowerCount(
                                    userId = userId,
                                    followerCount = count.dec()
                                )
                            }
                        }
                    })
            }
            .doOnError {
                if (AmityError.from(it) == AmityError.ITEM_NOT_FOUND) {
                    FollowLocalDataStore().updateFollowing(userId, AmityFollowStatus.NONE)
                } else {
                    FollowLocalDataStore().updateFollowing(userId, AmityFollowStatus.ACCEPTED)
                }
            }
    }

    fun getFollowingsPagingData(
        userId: String,
        filter: AmityFollowStatusFilter
    ): Flowable<PagingData<AmityFollowRelationship>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = false
            ),
            dynamicQueryStreamMediator = FollowingMediator(
                userId = userId,
                filter = filter
            ),
            pagingSourceFactory = {
                FollowLocalDataStore().getFollowingsPagingSource(
                    userId = userId,
                    filter = filter
                )
            },
            modelMapper = FollowRelationshipModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun getFollowersPagingData(
        userId: String,
        filter: AmityFollowStatusFilter
    ): Flowable<PagingData<AmityFollowRelationship>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = false
            ),
            dynamicQueryStreamMediator = FollowerMediator(
                userId = userId,
                filter = filter
            ),
            pagingSourceFactory = {
                FollowLocalDataStore().getFollowersPagingSource(
                    userId = userId,
                    filter = filter
                )
            },
            modelMapper = FollowRelationshipModelMapper()
        )

        return pagerCreator.create().toRx3()
    }

    fun getLatestFollowing(
        userId: String,
        filter: AmityFollowStatusFilter
    ): Flowable<AmityFollowRelationship> {
        return FollowLocalDataStore().getLatestFollowing(userId = userId, filter = filter)
            .map { FollowRelationshipModelMapper().map(it) }
    }

    fun getLatestFollower(
        userId: String,
        filter: AmityFollowStatusFilter
    ): Flowable<AmityFollowRelationship> {
        return FollowLocalDataStore().getLatestFollower(userId = userId, filter = filter)
            .map { FollowRelationshipModelMapper().map(it) }
    }

    fun blockUser(userId: String): Completable {
        return FollowRemoteDataStore().blockUser(userId)
            .flatMapCompletable {
                FollowQueryPersister().persist(it)
            }
    }

    fun unblockUser(userId: String): Completable {
        return FollowRemoteDataStore().unblockUser(userId)
            .flatMapCompletable {
                FollowQueryPersister().persist(it)
            }
    }

    fun getBlockedUsersPagingData(): Flowable<PagingData<AmityFollowRelationship>> {
        val userId = AmityCoreClient.getUserId()
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = false
            ),
            dynamicQueryStreamMediator = BlockedMediator(userId),
            pagingSourceFactory = {
                FollowLocalDataStore().getBlockedPagingSource(
                    userId = userId
                )
            },
            modelMapper = FollowRelationshipModelMapper()
        )

        return pagerCreator.create().toRx3()
    }

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }

}