package com.amity.socialcloud.sdk.social.data.storytarget

import androidx.room.Dao
import androidx.room.Query
import com.amity.socialcloud.sdk.api.social.storytarget.AmityGlobalStoryTargetsQueryOption
import com.amity.socialcloud.sdk.entity.social.story.StoryTargetEntity
import com.ekoapp.ekosdk.internal.data.dao.EkoObjectDao
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import org.joda.time.DateTime


@Dao
abstract class StoryTargetDao : EkoObjectDao<StoryTargetEntity>() {

    @Query("select * from storyTarget where uniqueId = :uniqueId")
    abstract fun observeStory(uniqueId: String): Flowable<StoryTargetEntity>

    @Query("delete from storyTarget")
    abstract override fun deleteAll()

    @Query("select * from storyTarget where uniqueId = :uniqueId")
    abstract fun getByIdNowImpl(uniqueId: String): StoryTargetEntity?

    override fun getByIdNow(uniqueId: String): StoryTargetEntity? {
        return getByIdNowImpl(uniqueId)
    }


    @Query("select * from storyTarget where targetType = :targetType and targetId = :targetId")
    abstract fun getStoryTarget(targetType: String, targetId: String): StoryTargetEntity?

    @Query("update storyTarget set localLastStoryExpiresAt = :localLastStoryExpiresAt where targetType = :targetType and targetId = :targetId")
    abstract fun updateStoryTargetLocalLastStoryExpiresAt(
        targetType: String,
        targetId: String,
        localLastStoryExpiresAt: DateTime?
    ): Completable

    @Query("update storyTarget set lastStoryExpiresAt = :lastStoryExpiresAt where targetType = :targetType and targetId = :targetId")
    abstract fun updateStoryTargetLastStoryExpiresAt(
        targetType: String,
        targetId: String,
        lastStoryExpiresAt: DateTime?
    ): Completable

    @Query("update storyTarget set localLastStorySeenExpiresAt = :localLastStorySeenExpiresAt where targetType = :targetType and targetId = :targetId")
    abstract fun updateStoryTargetLocalLastStorySeenExpiresAt(
        targetType: String,
        targetId: String,
        localLastStorySeenExpiresAt: DateTime?
    ): Completable


    @Query("update storyTarget set localSortingDate = :localSortingDate, hasUnseen = :hasUnseen where targetType = :targetType and targetId = :targetId")
    abstract fun updateStoryTargetHasUnseen(
        targetType: String,
        targetId: String,
        hasUnseen: Boolean,
        localSortingDate: DateTime?
    ): Completable

    @Query(
        "SELECT *" +
                " from storyTarget" +
                " where storyTarget.hasUnseen = (case when :hasUnseen is null then hasUnseen else :hasUnseen end)" +
                " and storyTarget.updatedAt > :now" +
                " and storyTarget.uniqueId not in " +
                "(" +
                "SELECT amity_paging_id.id" +
                " from amity_paging_id" +
                " where amity_paging_id.hash = (:hash)" +
                " and amity_paging_id.nonce = (:nonce) " +
                ")" +
                " order by storyTarget.updatedAt desc" +
                " limit 1"
    )
    abstract fun getLatestStoryTargetsImpl(
        hasUnseen: Boolean?,
        hash: Int,
        nonce: Int,
        now: DateTime?
    ): Flowable<StoryTargetEntity>

    fun getLatestStoryTargets(
        option: AmityGlobalStoryTargetsQueryOption,
        hash: Int,
        nonce: Int,
        now: DateTime?
    ): Flowable<StoryTargetEntity> {
        val hasUnseen = when(option) {
            AmityGlobalStoryTargetsQueryOption.UNSEEN -> true
            AmityGlobalStoryTargetsQueryOption.SEEN -> false
            else -> null
        }
        return getLatestStoryTargetsImpl(
            hasUnseen,
            hash,
            nonce,
            now
        )
    }

    @Query("select storyTarget.uniqueId from storyTarget" +
            " where storyTarget.hasUnseen = (case when :hasUnseen is null then hasUnseen else :hasUnseen end)" +
            " and storyTarget.localSortingDate > :expiresAfter" +
            " and storyTarget.localSortingDate = storyTarget.localLastStoryExpiresAt")
    abstract fun findCacheKeysImpl(
        hasUnseen: Boolean?,
        expiresAfter: DateTime?
    ): List<String>

    fun findCacheKeys(
        option: AmityGlobalStoryTargetsQueryOption,
        expiresAfter: DateTime?
    ): List<String> {
        return findCacheKeysImpl(
            hasUnseen = when(option) {
                AmityGlobalStoryTargetsQueryOption.SEEN -> false
                else -> true
            },
            expiresAfter = expiresAfter
        )
    }

}