package com.amity.socialcloud.sdk.model.core.error

import okhttp3.Response
import org.json.JSONObject
import retrofit2.HttpException
import java.io.IOException

open class AmityException protected constructor(
    message: String?,
    cause: Throwable?,
    val code: Int,
    internal val httpStatusCode: Int?
) : Exception(message, cause) {

    companion object {

        fun create(
            message: String?,
            cause: Throwable?,
            error: AmityError,
            httpStatusCode: Int? = null
        ): AmityException {
            return create(message, cause, error.code, httpStatusCode)
        }

        @JvmStatic
        fun create(
            message: String?,
            cause: Throwable?,
            code: Int,
            httpStatusCode: Int? = null
        ): AmityException {
            return AmityException(message, cause, code, httpStatusCode)
        }

        fun fromThrowable(throwable: Throwable): AmityException {
            // system error
            if (throwable is HttpException) {
                val errorBody = throwable.response()?.errorBody()?.string()
                if (errorBody != null) {
                    return try {
                        val error = JSONObject(errorBody)
                        val code = error.getInt("code")
                        val message = error.getString("message")
                        create(message, null, AmityError.from(code), throwable.code())
                    } catch (e: java.lang.Exception) {
                        create("corrupted error format", null, AmityError.UNKNOWN, throwable.code())
                    }
                }
            }

            // network error
            if (throwable is IOException) {
                return create("network error", null, AmityError.CONNECTION_ERROR)
            }
            
            if (throwable is AmityException) {
                return throwable
            }

            // unknown error
            return create("unknown error", null, AmityError.UNKNOWN)
        }

        fun fromResponse(response: Response): AmityException {
            return try {
                /*
                https://github.com/square/retrofit/issues/3336
                avoid response being read and closed before being used by Retrofit internals by using peekBody()
                */
                val errorBody = response.peekBody(Long.MAX_VALUE).string()
                val error = JSONObject(errorBody)
                val code = error.getInt("code")
                val message = error.getString("message")
                create(message, null, AmityError.from(code))
            } catch (e: java.lang.Exception) {
                create("corrupted error format", null, AmityError.UNKNOWN)
            }
        }
    }

}