package com.amity.socialcloud.sdk.model.social.story

import android.os.Parcelable
import com.amity.socialcloud.sdk.core.JsonObjectParceler
import com.amity.socialcloud.sdk.model.core.file.AmityImage
import com.amity.socialcloud.sdk.model.core.file.AmityVideo
import com.amity.socialcloud.sdk.model.core.reaction.AmityReactionMap
import com.amity.socialcloud.sdk.model.core.user.AmityUser
import com.ekoapp.ekosdk.JsonArrayParceler
import com.ekoapp.ekosdk.ReactorObject
import com.google.common.base.Objects
import com.google.gson.JsonArray
import com.google.gson.JsonObject
import kotlinx.parcelize.Parcelize
import kotlinx.parcelize.TypeParceler
import org.joda.time.DateTime


@Parcelize
@TypeParceler<JsonObject?, JsonObjectParceler>
@TypeParceler<JsonArray?, JsonArrayParceler>
data class AmityStory internal constructor(
    private val storyId: String,
    private val uniqueId: String,
    private val targetType: TargetType,
    private val targetId: String,
    internal var target: AmityStoryTarget? = null,
    private val dataType: DataType,
    internal var data: Data = Data.UNKNOWN,
    internal val rawData: JsonObject?,
    internal var storyItems: List<AmityStoryItem> = emptyList(),
    internal val rawStoryItems: JsonArray?,
    private val metadata: JsonObject?,
    private val state: State,
    internal var myReactions: List<String> = emptyList(),
    private val reactions: AmityReactionMap,
    private val reactionCount: Int,
    private val commentCount: Int,
    private val flagCount: Int,
    private val isDeleted: Boolean,
    private val expiresAt: DateTime?,
    private val createdAt: DateTime?,
    private val updatedAt: DateTime?,
    internal val path: String,
    private val impression: Int,
    private val reach: Int,
    internal var creator: AmityUser? = null,
    private val creatorId: String
) : Parcelable, ReactorObject {

    fun getUniqueId(): String {
        return uniqueId
    }

    fun getStoryId(): String {
        return storyId
    }

    fun getTargetType(): TargetType {
        return targetType
    }

    fun getTargetId(): String {
        return targetId
    }

    fun getTarget(): AmityStoryTarget? {
        return target
    }

    fun getState(): State {
        return state
    }

    fun getDataType(): DataType {
        return dataType
    }

    fun getData(): Data {
        return data
    }

    fun getStoryItems(): List<AmityStoryItem> {
        return storyItems
    }

    fun getMetadata(): JsonObject? {
        return metadata
    }

    fun getMyReactions(): List<String> {
        return myReactions
    }

    fun getReactions(): AmityReactionMap {
        return reactions
    }

    internal fun getFlagCount(): Int {
        return flagCount
    }

    fun getReactionCount(): Int {
        return reactionCount
    }

    fun getCommentCount(): Int {
        return commentCount
    }

    fun getImpression(): Int {
        return impression
    }

    fun getReach(): Int {
        return reach
    }

    fun isDeleted(): Boolean {
        return isDeleted
    }

    fun getCreatorId(): String {
        return creatorId
    }

    fun getCreator(): AmityUser? {
        return creator
    }

    fun getExpiresAt(): DateTime? {
        return expiresAt
    }

    fun getCreatedAt(): DateTime? {
        return createdAt
    }

    fun isSeen(): Boolean {
        val lastStorySeenExpiresAt = target?.highestSeen
        return if(lastStorySeenExpiresAt == null) {
            false
        } else {
            lastStorySeenExpiresAt.isEqual(expiresAt) || lastStorySeenExpiresAt.isAfter(expiresAt)
        }
    }

    sealed class Data : Parcelable {

        @Parcelize
        @TypeParceler<JsonObject?, JsonObjectParceler>
        class IMAGE(
            internal val storyId: String,
            internal val rawData: JsonObject?,
            internal val imageDisplayMode: AmityStoryImageDisplayMode,
            internal var image: AmityImage? = null
        ) : Data() {

            fun getStoryId(): String {
                return storyId
            }

            fun getImage(): AmityImage? {
                return image
            }

            fun getImageDisplayMode(): AmityStoryImageDisplayMode {
                return imageDisplayMode
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is IMAGE
                        && Objects.equal(other.storyId, storyId)
                        && Objects.equal(other.rawData, rawData))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(storyId, rawData)
            }

        }

        @Parcelize
        @TypeParceler<JsonObject?, JsonObjectParceler>
        class VIDEO(
            internal val storyId: String,
            internal val rawData: JsonObject?,
            internal var thumbnail: AmityImage? = null,
            internal var video: AmityVideo? = null
        ) : Data() {

            fun getStoryId(): String {
                return storyId
            }

            fun getThumbnailImage(): AmityImage? {
                return thumbnail
            }

            fun getVideo(): AmityVideo? {
                return video
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is VIDEO
                        && Objects.equal(other.storyId, storyId)
                        && Objects.equal(other.rawData, rawData))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(storyId, rawData)
            }
        }

        @Parcelize
        object UNKNOWN : Data()

    }

    enum class DataType(val apiKey: String) {
        IMAGE("image"),
        VIDEO("video"),
        UNKNOWN("unknown");

        companion object {
            fun enumOf(value: String?): DataType {
                return values().find { it.apiKey == value } ?: UNKNOWN
            }
        }
    }

    enum class TargetType(val apiKey: String) {
        //USER("user"),
        COMMUNITY("community"),
        UNKNOWN("unknown");

        companion object {
            fun enumOf(value: String?): TargetType {
                return values().find { it.apiKey == value } ?: UNKNOWN
            }
        }
    }

    enum class State(val apiKey: String) {
        SYNCING("syncing"),
        SYNCED("synced"),
        FAILED("failed");

        companion object {
            fun enumOf(value: String): State = values().find { it.apiKey == value } ?: SYNCED
        }
    }

    override fun updatedAt(): DateTime? {
        return updatedAt
    }

    override fun uniqueId(): String {
        return uniqueId
    }

}