package com.amity.socialcloud.sdk.core

import com.amity.socialcloud.sdk.api.core.AmityCoreClient
import com.amity.socialcloud.sdk.chat.data.channel.ChannelRepository
import com.amity.socialcloud.sdk.chat.data.marker.subchannel.SubChannelMarkerRepository
import com.amity.socialcloud.sdk.chat.data.subchannel.SubChannelRepository
import com.amity.socialcloud.sdk.chat.domain.marker.channel.ReCalculateChannelUnreadInfoUseCase
import com.amity.socialcloud.sdk.chat.domain.marker.subchannel.OptimisticCreateSubChannelUnreadInfoUseCase
import com.amity.socialcloud.sdk.chat.domain.marker.sync.SyncMarkerUseCase
import com.amity.socialcloud.sdk.chat.domain.marker.user.GetUserMarkerUseCase
import com.amity.socialcloud.sdk.core.mention.AmityMentionType
import com.amity.socialcloud.sdk.core.session.component.SessionComponent
import com.amity.socialcloud.sdk.core.session.eventbus.MarkerEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.amity.socialcloud.sdk.log.AmityLog
import com.amity.socialcloud.sdk.model.chat.channel.AmityChannel
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.disposables.CompositeDisposable
import io.reactivex.rxjava3.schedulers.Schedulers
import org.joda.time.DateTime
import java.util.concurrent.TimeUnit

internal class MarkerSyncEngine(
	sessionLifeCycleEventBus: SessionLifeCycleEventBus,
	sessionStateEventBus: SessionStateEventBus
) : SessionComponent(sessionLifeCycleEventBus, sessionStateEventBus) {
	
	private val timer = Flowable.interval(SYNC_INTERVAL,TimeUnit.SECONDS)
			.flatMapCompletable {
				timerDidTrigger().onErrorComplete()
			}
			.subscribeOn(Schedulers.io())
	
	private var isWaitingForResponse: Boolean = false
	private val events: MutableList<MarkerSyncEvent> = mutableListOf()
	private var deviceLastSyncAt: DateTime? = null
	private var isReadyToSync: Boolean = false
	private var isConsistentMode = true
	
	private val disposable = CompositeDisposable()
	
	init {
		MarkerEventBus
			.observe()
			.doOnNext { event ->
				when (event) {
					is MarkerEvent.NetworkConnection.Connected -> {
						addNewEvent(MarkerSyncEvent.RESUME)
					}
					is MarkerEvent.NetworkConnection.Disconnected -> {
						isWaitingForResponse = false
					}
					is MarkerEvent.NewMessage -> {
						if (event.message.channelType.let(::isMarkerSyncSupport)) {
							addNewEvent(MarkerSyncEvent.NEW_MESSAGE)
							handleConsistentEvent(event)
						}
					}
					is MarkerEvent.MarkerUpdated -> {
						addNewEvent(MarkerSyncEvent.MARKER_UPDATED)
						handleConsistentEvent(event)
					}
					else -> {}
				}
			}
			.subscribeOn(Schedulers.io())
			.observeOn(Schedulers.io())
			.subscribe()
	}
	
	fun isConsistentMode(): Boolean {
		return this.isConsistentMode
	}
	
	fun startMarkerSync() {
		CoreClient.enableUnreadCount()
		isConsistentMode = false
		addNewEvent(MarkerSyncEvent.START_SYNCING)
		if (disposable.size() == 0 || disposable.isDisposed){
			disposable.clear()
			timer.subscribe().let(disposable::add)
		}
	}
	
	fun startUserTotalUnreadSync() {
		addNewEvent(MarkerSyncEvent.START_SYNCING)
		if (disposable.size() == 0 || disposable.isDisposed){
			disposable.clear()
			timer.subscribe().let(disposable::add)
		}
	}
	
	fun stopMarkerSync() {
		if (disposable.size() > 0){
			disposable.clear()
			events.clear()
		}
		isWaitingForResponse = false
	}
	
	private fun timerDidTrigger(): Completable {
		return when {
			isWaitingForResponse -> Completable.complete()
			events.isEmpty() -> Completable.complete()
			!isReadyToSync -> Completable.complete()
			else -> {
				events.clear()
				isWaitingForResponse = true
				fetchDeviceLastSyncAt()
					.flatMap {
						SyncMarkerUseCase().execute(getDeviceLastSyncAt())
					}
						.doOnSuccess { (userMarkers, hasMoreMarkers) ->
							userMarkers.firstOrNull()?.lastSyncAt?.let(::saveDeviceLastSyncAt)
							if (hasMoreMarkers) {
								addNewEvent(MarkerSyncEvent.HAS_MORE)
							}
						}
						.doFinally {
							isWaitingForResponse = false
						}
						.ignoreElement()
						.subscribeOn(Schedulers.io())
			}
		}
	}
	
	private fun addNewEvent(event : MarkerSyncEvent){
		events.add(event)
	}
	
	private fun fetchDeviceLastSyncAt(): Single<Boolean> {
		return if (deviceLastSyncAt == null) {
			GetUserMarkerUseCase().execute()
				.doOnSuccess { userMarkers ->
					userMarkers.userMarkers.firstOrNull()?.let { userMarker ->
						saveDeviceLastSyncAt(userMarker.lastSyncAt)
					}
				}
				.map { true }
				.onErrorReturnItem(true)
		} else {
			Single.just(true)
		}
	}
	
	private fun getDeviceLastSyncAt(): DateTime? {
		return deviceLastSyncAt
	}
	
	private fun saveDeviceLastSyncAt(lastSyncAt: DateTime){
		if (deviceLastSyncAt == null || deviceLastSyncAt?.isBefore(lastSyncAt) == true) {
			deviceLastSyncAt = lastSyncAt
		}
	}
	
	private fun handleConsistentEvent(event: MarkerEvent) {
		when (event) {
			is MarkerEvent.SubChannelCreated -> {
				val subChannel = event.dto.subChannels.firstOrNull()
				val subChannelId = subChannel?.subChannelId
				val channelId = subChannel?.channelId
				if (subChannelId != null && channelId != null) {
					OptimisticCreateSubChannelUnreadInfoUseCase().execute(subChannelId, channelId)
				}
			}
			is MarkerEvent.SubChannelDeleted -> {
				event.dto.subChannels.firstOrNull()?.let { subChannel ->
					subChannel.subChannelId?.let {
						SubChannelMarkerRepository().deleteUnreadInfoBySubChannelId(it)
					}
					subChannel.channelId?.let {
						ReCalculateChannelUnreadInfoUseCase().execute(it)
						ChannelRepository().notifyChanges(it)
					}
				}
			}
			is MarkerEvent.ChannelLeft,
			is MarkerEvent.ChannelDeleted,
			is MarkerEvent.ChannelBan -> {
				(event as? MarkerEvent.Channel)?.dto?.channelDtoList?.firstOrNull()?.channelId?.let {
					SubChannelMarkerRepository().deleteUnreadInfoByChannelId(it)
					ReCalculateChannelUnreadInfoUseCase().execute(it)
				}
			}
			is MarkerEvent.NewMessage -> {
				val subChannelId = event.message.subChannelId
				val channelId = event.message.channelId
				val subChannelUnreadInfo = SubChannelMarkerRepository().getSubChannelUnreadInfo(subChannelId)
				if (subChannelUnreadInfo != null && event.message.segment > subChannelUnreadInfo.lastSegment) {
					subChannelUnreadInfo.lastSegment = event.message.segment
					val isMentionedAll = event.message.mentionees
						.mapNotNull { it.type }
						.any { it == AmityMentionType.CHANNEL.apiKey }
					val isMentioned = if (isMentionedAll) {
						true
					} else {
						event.message.mentionees
							.mapNotNull { it.userPublicIds }
							.flatten()
							.contains(AmityCoreClient.getUserId())
					}
					if (isMentioned) {
						subChannelUnreadInfo.lastMentionSegment = event.message.segment
					}
					subChannelUnreadInfo.updatedAt = DateTime.now()
					SubChannelMarkerRepository().saveSubChannelUnreadInfo(listOf(subChannelUnreadInfo))
					SubChannelRepository().notifyChanges(subChannelId)
					ChannelRepository().notifyChanges(channelId)
					// Mark read the message immediately if it's sent by the current user
					if (event.message.userId == AmityCoreClient.getUserId()) {
						CoreClient.markRead(event.message.subChannelId, event.message.segment)
					}
				}
			}
			else -> {}
		}
	}
	
	companion object {
		private const val SYNC_INTERVAL = 2L
		private val SUPPORT_CHANNEL_TYPES = listOf(AmityChannel.Type.COMMUNITY,AmityChannel.Type.CONVERSATION,AmityChannel.Type.BROADCAST)
		
		fun isMarkerSyncSupport(type: AmityChannel.Type?): Boolean {
			return type?.let(SUPPORT_CHANNEL_TYPES::contains) ?: false
		}
		
		fun isMarkerSyncSupport(type: String?): Boolean {
			return type?.let(SUPPORT_CHANNEL_TYPES.map{it.apiKey}::contains) ?: false
		}
	}
	
	override fun onSessionStateChange(sessionState: SessionState) {
		when(sessionState) {
			SessionState.Established -> {
				isReadyToSync = true			}
			else -> {
				isReadyToSync = false
			}
		}
	}
	
	override fun establish(account: EkoAccount) {
		isReadyToSync = true
		if (CoreClient.isUnreadCountEnable() && isConsistentMode) {
			startUserTotalUnreadSync()
		}
	}
	
	override fun destroy() {
		stopMarkerSync()
		isReadyToSync = false
		deviceLastSyncAt = null
	}
	
	override fun handleTokenExpire() {
		stopMarkerSync()
		isReadyToSync = false
	}
}