package com.amity.socialcloud.sdk.core.data.reaction.paging

import androidx.paging.ExperimentalPagingApi
import co.amity.rxbridge.toRx2
import com.amity.socialcloud.sdk.core.data.reaction.ReactionQueryPersister
import com.amity.socialcloud.sdk.core.data.reaction.ReactionRemoteDataStore
import com.amity.socialcloud.sdk.model.core.reaction.AmityReaction
import com.amity.socialcloud.sdk.model.core.reaction.AmityReactionReferenceType
import com.ekoapp.ekosdk.internal.api.dto.EkoReactionAndUserListDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.data.model.EkoReactionEntity
import com.ekoapp.ekosdk.internal.keycreator.ReactionKeyCreator
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import com.ekoapp.ekosdk.internal.token.QueryStreamQueryToken
import com.ekoapp.ekosdk.internal.usecase.reaction.ReactionReactorUseCase
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single


@OptIn(ExperimentalPagingApi::class)
class ReactionMediator(
    val referenceType: AmityReactionReferenceType,
    val referenceId: String,
    val reactionName: String?
) :
    DynamicQueryStreamMediator<EkoReactionEntity, EkoReactionAndUserListDto, AmityReaction>(
        dynamicQueryStreamKeyCreator = ReactionKeyCreator(
            referenceType = referenceType,
            referenceId = referenceId,
            reactionName = reactionName
        ),
        nonce = AmityNonce.REACTION_LIST
    ) {

    override fun provideReactorPublisher(): Flowable<AmityReaction> {
        return ReactionReactorUseCase().execute(
            referenceType = referenceType,
            referenceId = referenceId,
            reactionName = reactionName
        ).toRx2()
    }

    override fun getFirstPageRequest(pageSize: Int): Single<EkoReactionAndUserListDto> {
        return getRequest(
            reactionName = reactionName,
            limit = pageSize
        )
    }

    override fun getFetchByTokenRequest(token: String): Single<EkoReactionAndUserListDto> {
        return getRequest(
            reactionName = reactionName,
            token = token
        )
    }

    override fun persistResponse(dto: EkoReactionAndUserListDto): Completable {
        return ReactionQueryPersister(
            referenceType = referenceType,
            referenceId = referenceId
        ).persist(dto).toRx2()
    }

    override fun convertResponseToQueryToken(dto: EkoReactionAndUserListDto): QueryStreamQueryToken {
        val reactionIds = mutableListOf<String>()
        dto.reactions.forEach { reactionDto ->
            reactionDto.reactors.forEach { reactorDto ->
                reactionIds.add(reactorDto.reactionId)
            }
        }
        return DynamicQueryStreamQueryToken(
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            next = dto.token?.next,
            previous = dto.token?.previous,
            primaryKeys = reactionIds
        )
    }

    private fun getRequest(
        reactionName: String?,
        limit: Int? = null,
        token: String? = null
    ): Single<EkoReactionAndUserListDto> {
        return ReactionRemoteDataStore().getReactions(
            referenceId = referenceId,
            referenceType = referenceType,
            reactionName = reactionName,
            limit = limit,
            token = token
        ).toRx2()
    }
}