package com.amity.socialcloud.sdk.video.presentation

import android.content.Context
import android.net.Uri
import android.text.format.DateFormat
import android.util.AttributeSet
import android.view.View
import android.widget.LinearLayout
import com.amity.socialcloud.sdk.streamapi.PartialStreamData
import com.amity.socialcloud.sdk.video.StreamPlayerClient
import com.ekoapp.sdk.streamplayer.R
import com.google.android.exoplayer2.C
import com.google.android.exoplayer2.ExoPlayer
import com.google.android.exoplayer2.MediaItem
import com.google.android.exoplayer2.Player
import com.google.android.exoplayer2.audio.AudioAttributes
import com.google.android.exoplayer2.extractor.DefaultExtractorsFactory
import com.google.android.exoplayer2.source.MediaSource
import com.google.android.exoplayer2.source.ProgressiveMediaSource
import com.google.android.exoplayer2.ui.StyledPlayerView
import com.google.android.exoplayer2.upstream.DataSource
import com.google.android.exoplayer2.upstream.DefaultDataSourceFactory
import com.google.android.exoplayer2.util.Util
import io.reactivex.rxjava3.android.schedulers.AndroidSchedulers
import io.reactivex.rxjava3.disposables.Disposable
import io.reactivex.rxjava3.functions.Consumer
import io.reactivex.rxjava3.internal.operators.flowable.FlowableInterval
import io.reactivex.rxjava3.schedulers.Schedulers
import kotlinx.android.synthetic.main.view_amity_video_player.view.*
import org.joda.time.DateTime
import java.util.concurrent.TimeUnit


class AmityVideoPlayer : LinearLayout {

    private lateinit var playerView: StyledPlayerView
    private var exoplayer: ExoPlayer? = null
    private var isStopped = false
    private var isVideoPlaying = false
    private var isStopWhenPause = false
    private var startedAt = DateTime.now()
    private var duration = 0L
    private var durationDisposable: Disposable? = null
    private var title = ""
    private var sessionId: String? = null


    constructor(context: Context) : super(context) {
        inflateView()
    }

    constructor(context: Context, attrs: AttributeSet) : super(context, attrs) {
        inflateView()
    }

    constructor(context: Context, attrs: AttributeSet, defStyleAttr: Int) : super(context, attrs, defStyleAttr) {
        inflateView()
    }

    private fun inflateView() {
        View.inflate(context, R.layout.view_amity_video_player, this)
        playerView = findViewById(R.id.amity_video_viewer);
    }

    fun play(streamId: String, url: String) {
        initPlayer()
        prepareVideo(url)
        prepareCounting(streamId)
    }

    fun stop() {
        exoplayer?.stop(true)
        stopCounting()
    }

    fun pause() {
        exoplayer?.playWhenReady = false
        exoplayer?.playbackState
    }

    fun resume() {
        exoplayer?.playWhenReady = true
        exoplayer?.playbackState
    }

    fun enableStopWhenPause() {
        isStopWhenPause = true
    }

    private fun initPlayer() {
        exoplayer = ExoPlayer.Builder(context).build()
        exoplayer?.playWhenReady = true
        exoplayer?.addListener(object : Player.Listener {
            override fun onIsPlayingChanged(isPlaying: Boolean) {
                isVideoPlaying = isPlaying
                if (isStopWhenPause) {
                    seekToLastWhenResume(isPlaying)
                }
            }
        })
        AudioAttributes.Builder()
                .setUsage(C.USAGE_MEDIA)
                .setContentType(C.AUDIO_CONTENT_TYPE_MOVIE)
                .build()
                .let { audioAttributes ->
                    exoplayer?.setAudioAttributes(audioAttributes, true)
                }
        playerView.apply {
            player = exoplayer
        }
        //amity_video_viewer.player = exoplayer
    }

    private fun seekToLastWhenResume(isPlaying: Boolean) {
        if (isPlaying && exoplayer?.playbackState == Player.STATE_READY) {
            if (isStopped) {
                exoplayer?.seekTo(Long.MAX_VALUE)
                isStopped = false
            }
        } else if (!isPlaying && exoplayer?.playbackState == Player.STATE_READY) {
            isStopped = true
        }
    }

    private fun prepareVideo(url: String) {
        val newUrl = url.replace("https", "http")
        amity_video_viewer.requestFocus()
        val mediaItem = MediaItem.fromUri(Uri.parse(newUrl))
        val videoSource: MediaSource = ProgressiveMediaSource.Factory(getDataSourceFactory(), DefaultExtractorsFactory())
                .createMediaSource(mediaItem)
        exoplayer?.prepare(videoSource)
    }

    private fun getDataSourceFactory(): DataSource.Factory {
        return DefaultDataSourceFactory(context, Util.getUserAgent(context, this.javaClass.simpleName))
    }

    private fun prepareCounting(streamId: String) {
        if (durationDisposable == null) {
            StreamPlayerClient.getFunction().getStreamData(streamId)
                    .subscribeOn(Schedulers.io())
                    .doOnSuccess {
                        if (it.isLive) {
                            starCounting(it)
                        }
                    }
                    .subscribe()
        }
    }

    private fun starCounting(partialStreamData: PartialStreamData) {
        StreamPlayerClient.getFunction().createStreamSession(partialStreamData.streamId,
                title, startedAt, partialStreamData.resolution)
                .subscribeOn(Schedulers.io())
                .subscribe(Consumer {
                    sessionId = it
                    startInterval()
                })
    }

    private fun startInterval() {
        startedAt = DateTime.now()
        durationDisposable = FlowableInterval(0, 1, TimeUnit.SECONDS, Schedulers.io())
                .filter { isVideoPlaying }
                .map { duration++ }
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .doOnNext {
                    val endedAt = DateTime.now()
                    updateStreamSession(it, endedAt)
                }
                .subscribe()
    }

    private fun stopCounting() {
        if (durationDisposable?.isDisposed == false) {
            durationDisposable?.dispose()
            if (duration > 0L) {
                val endedAt = DateTime.now()
                val startTimeText = DateFormat.format("MMMM dd yyyy, h:mm aa", startedAt.millis).toString()
                val endTimeText = DateFormat.format("MMMM dd yyyy, h:mm aa", endedAt.millis).toString()
                val summaryText = " Start time : $startTimeText\n End time : $endTimeText\n Duration in second : $duration"
                updateStreamSession(duration, endedAt)
                syncPendingSession()
            }
        }
    }

    private fun updateStreamSession(duration: Long?, endedAt: DateTime?) {
        sessionId?.let {
            StreamPlayerClient.getFunction().updateStreamSession(it, duration, endedAt)
                    .subscribeOn(Schedulers.io())
                    .subscribe()
        }
    }

    private fun syncPendingSession() {
        StreamPlayerClient.getFunction().syncPendingSession()
    }
}