package com.amity.socialcloud.sdk.video.listener

import android.content.Context
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.os.Build
import com.amity.socialcloud.sdk.video.AmityStreamBroadcaster
import com.amity.socialcloud.sdk.video.model.AmityStreamBroadcasterState
import com.amity.github.faucamp.simplertmp.RtmpHandler
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.android.schedulers.AndroidSchedulers
import io.reactivex.rxjava3.disposables.Disposable
import io.reactivex.rxjava3.internal.operators.single.SingleTimer
import io.reactivex.rxjava3.subjects.PublishSubject
import java.io.IOException
import java.net.SocketException
import java.util.concurrent.TimeUnit

class EkoRtmpListener(val context: Context, private val broadcaster: AmityStreamBroadcaster,
                      private val statusSubject: PublishSubject<AmityStreamBroadcasterState>) : RtmpHandler.RtmpListener {

    var currentStatus: AmityStreamBroadcasterState? = AmityStreamBroadcasterState.CONNECTING()
    var isReconnecting = false
    var isPublishing = false
    var reconnectionDisposable: Disposable? = null

    private fun reconnect() {
        if (!isReconnecting) {
            isReconnecting = true
            statusSubject.onNext(AmityStreamBroadcasterState.CONNECTING())
            reconnectionDisposable = Flowable.interval(1, TimeUnit.SECONDS)
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe({ tryResumeBroadcast() }, { })
        }
    }

    private fun tryResumeBroadcast() {
        if (currentStatus is AmityStreamBroadcasterState.DISCONNECTED && isOnline(context) && !isPublishing) {
            isPublishing = true
            SingleTimer(1, TimeUnit.SECONDS, AndroidSchedulers.mainThread())
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe({ broadcaster.resumePublish() }, { })
        }
    }

    fun isOnline(context: Context): Boolean {
        val cm = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            val n = cm.activeNetwork
            if (n != null) {
                val nc = cm.getNetworkCapabilities(n)
                //It will check for both wifi and cellular network
                return nc!!.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR) || nc.hasTransport(NetworkCapabilities.TRANSPORT_WIFI)
            }
            return false
        } else {
            val netInfo = cm.activeNetworkInfo
            return netInfo != null && netInfo.isConnectedOrConnecting
        }
    }

    override fun onRtmpConnecting(msg: String?) {
        statusSubject.onNext(AmityStreamBroadcasterState.CONNECTING())
    }

    override fun onRtmpConnected(msg: String?) {
        currentStatus = AmityStreamBroadcasterState.CONNECTED()
        statusSubject.onNext(AmityStreamBroadcasterState.CONNECTED())
        isReconnecting = false
        isPublishing = false
        if (reconnectionDisposable?.isDisposed == false) {
            reconnectionDisposable?.dispose()
        }
    }

    override fun onRtmpVideoStreaming() {}

    override fun onRtmpAudioStreaming() {}

    override fun onRtmpStopped() {}

    override fun onRtmpDisconnected() {}

    override fun onRtmpVideoFpsChanged(fps: Double) {}

    override fun onRtmpVideoBitrateChanged(bitrate: Double) {}

    override fun onRtmpAudioBitrateChanged(bitrate: Double) {}

    override fun onRtmpSocketException(e: SocketException?) {
        broadcaster.pausePublish()
        broadcaster.startPreview()
        currentStatus = AmityStreamBroadcasterState.DISCONNECTED()
        statusSubject.onNext(AmityStreamBroadcasterState.DISCONNECTED())
        reconnect()
    }

    override fun onRtmpIOException(e: IOException?) {
        statusSubject.onNext(AmityStreamBroadcasterState.DISCONNECTED())
    }

    override fun onRtmpIllegalArgumentException(e: IllegalArgumentException?) {
        statusSubject.onNext(AmityStreamBroadcasterState.DISCONNECTED())
    }

    override fun onRtmpIllegalStateException(e: IllegalStateException?) {
        statusSubject.onNext(AmityStreamBroadcasterState.DISCONNECTED())
    }
}
