package co.androidian.network.utils;

import java.net.UnknownHostException;
import java.util.concurrent.TimeUnit;

import io.reactivex.functions.BiFunction;
import io.reactivex.functions.Function;
import rx.Observable;
import rx.functions.Func1;
import rx.functions.Func2;

/**
 * <p>RxJava and RxJava2 combined retry module for retrofit2.
 * It can be added to any request chain or subscription.
 * Created on 13/09/16.
 *
 * @author Omkar Todkar.
 */
public class RetryWithDelay implements Func1<Observable<? extends Throwable>, Observable<?>>,
        Function<io.reactivex.Observable<? extends Throwable>, io.reactivex.Observable<?>> {

    private final int MAX_RETRIES;

    private final int DELAY_DURATION;

    private final int START_RETRY;

    /**
     * Provide number of retries and seconds to be delayed between retry.
     *
     * @param maxRetries             Number of retries.
     * @param delayDurationInSeconds Seconds to be delays in each retry.
     */
    public RetryWithDelay(int maxRetries, int delayDurationInSeconds) {
        MAX_RETRIES = maxRetries;
        DELAY_DURATION = delayDurationInSeconds;
        START_RETRY = 1;
    }

    @Override
    public Observable<?> call(Observable<? extends Throwable> observable) {
        return observable.delay(DELAY_DURATION, TimeUnit.SECONDS)
                .zipWith(Observable.range(START_RETRY, MAX_RETRIES),
                        new Func2<Throwable, Integer, Integer>() {
                            @Override
                            public Integer call(Throwable throwable, Integer attempt) {
                                if (throwable instanceof UnknownHostException) {
                                    attempt = MAX_RETRIES;
                                }
                                return attempt;
                            }
                        });
    }

    @Override
    public io.reactivex.Observable<?> apply(io.reactivex.Observable<? extends Throwable> observable) throws Exception {
        return observable.delay(DELAY_DURATION, TimeUnit.SECONDS)
                .zipWith(io.reactivex.Observable.range(START_RETRY, MAX_RETRIES),
                        new BiFunction<Throwable, Integer, Integer>() {
                            @Override
                            public Integer apply(Throwable throwable, Integer attempt) throws Exception {
                                if (throwable instanceof UnknownHostException) {
                                    attempt = MAX_RETRIES;
                                }
                                return attempt;
                            }
                        });
    }
}
