/*
 * Copyright © 2015 Cask Data, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package co.cask.cdap.data2.transaction.queue.hbase;

import co.cask.cdap.data2.queue.ConsumerConfig;
import co.cask.cdap.data2.queue.ConsumerGroupConfig;
import co.cask.cdap.data2.queue.QueueEntry;
import co.cask.cdap.data2.transaction.queue.QueueScanner;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.client.Scan;

import java.io.Closeable;
import java.io.IOException;
import java.util.Collection;

/**
 * Interface to represent the strategy being used in queue.
 */
public interface HBaseQueueStrategy extends Closeable {

  /**
   * Creates a {@link QueueScanner} from the given {@link Scan} on the HBase table.
   *
   * @param hTable  HTable for talking to HBase
   * @param scan    The scan request
   * @param numRows Maximum number of rows to scan for
   * @return A {@link QueueScanner} that scans over the give table
   */
  QueueScanner createScanner(ConsumerConfig consumerConfig, HTable hTable, Scan scan, int numRows) throws IOException;

  /**
   * Creates the actual row key used for accessing the HBase table from the given queue entry row key.
   */
  byte[] getActualRowKey(ConsumerConfig consumerConfig, byte[] originalRowKey);

  /**
   * Get all the row keys that the queue entries need to write to.
   *
   * @param consumerGroupConfigs Consumer groups' configurations for the queue
   * @param queueEntry Entry to enqueue
   * @param rowKeyPrefix Prefix for row keys
   * @param writePointer The writer pointer of the current transaction
   * @param counter The counter of the given entry in this transaction
   * @param rowKeys Collection for storing all the row keys generated by this entry
   */
  void getRowKeys(Iterable<ConsumerGroupConfig> consumerGroupConfigs, QueueEntry queueEntry, byte[] rowKeyPrefix,
                  long writePointer, int counter, Collection<byte[]> rowKeys);
}
