/*
 * Copyright 2014 - 2020 Rafael Winterhalter
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package co.elastic.apm.agent.shaded.bytebuddy.matcher;

import co.elastic.apm.agent.shaded.bytebuddy.build.HashCodeAndEqualsPlugin;
import co.elastic.apm.agent.shaded.bytebuddy.description.type.TypeDefinition;
import co.elastic.apm.agent.shaded.bytebuddy.description.type.TypeDescription;

/**
 * An element matcher that matches a super class.
 *
 * @param <T> The type of the matched entity.
 */
@HashCodeAndEqualsPlugin.Enhance
public class HasSuperClassMatcher<T extends TypeDescription> extends ElementMatcher.Junction.AbstractBase<T> {

    /**
     * The matcher to apply to any super class of the matched type.
     */
    private final ElementMatcher<? super TypeDescription.Generic> matcher;

    /**
     * Creates a new matcher for a super class.
     *
     * @param matcher The matcher to apply to any super class of the matched type.
     */
    public HasSuperClassMatcher(ElementMatcher<? super TypeDescription.Generic> matcher) {
        this.matcher = matcher;
    }

    /**
     * {@inheritDoc}
     */
    public boolean matches(T target) {
        if (target.isInterface()) {
            return matcher.matches(TypeDescription.Generic.OBJECT);
        }
        for (TypeDefinition typeDefinition : target) {
            if (matcher.matches(typeDefinition.asGenericType())) {
                return true;
            }
        }
        return false;
    }

    @Override
    public String toString() {
        return "hasSuperClass(" + matcher + ")";
    }
}
