/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache license, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the license for the specific language governing permissions and
 * limitations under the license.
 */

package co.elastic.apm.agent.shaded.apache.logging.log4j.core.config;

import java.util.Objects;

import co.elastic.apm.agent.shaded.apache.logging.log4j.Level;
import co.elastic.apm.agent.shaded.apache.logging.log4j.Marker;
import co.elastic.apm.agent.shaded.apache.logging.log4j.core.LogEvent;
import co.elastic.apm.agent.shaded.apache.logging.log4j.message.Message;
import co.elastic.apm.agent.shaded.apache.logging.log4j.status.StatusLogger;
import co.elastic.apm.agent.shaded.apache.logging.log4j.util.PropertiesUtil;
import co.elastic.apm.agent.shaded.apache.logging.log4j.util.Supplier;

/**
 * Reliability strategy that sleeps unconditionally for some time before allowing a Configuration to be stopped.
 */
public class AwaitUnconditionallyReliabilityStrategy implements ReliabilityStrategy, LocationAwareReliabilityStrategy {

    private static final long DEFAULT_SLEEP_MILLIS = 5000; // 5 seconds
    private static final long SLEEP_MILLIS = sleepMillis();
    private final LoggerConfig loggerConfig;

    public AwaitUnconditionallyReliabilityStrategy(final LoggerConfig loggerConfig) {
        this.loggerConfig = Objects.requireNonNull(loggerConfig, "loggerConfig is null");
    }

    private static long sleepMillis() {
        return PropertiesUtil.getProperties().getLongProperty("log4j.waitMillisBeforeStopOldConfig",
                DEFAULT_SLEEP_MILLIS);
    }

    /*
     * (non-Javadoc)
     *
     * @see co.elastic.apm.agent.shaded.apache.logging.log4j.core.config.ReliabilityStrategy#log(co.elastic.apm.agent.shaded.apache.logging.log4j.util.Supplier,
     * java.lang.String, java.lang.String, co.elastic.apm.agent.shaded.apache.logging.log4j.Marker, co.elastic.apm.agent.shaded.apache.logging.log4j.Level,
     * co.elastic.apm.agent.shaded.apache.logging.log4j.message.Message, java.lang.Throwable)
     */
    @Override
    public void log(final Supplier<LoggerConfig> reconfigured, final String loggerName, final String fqcn, final Marker marker, final Level level,
            final Message data, final Throwable t) {
        loggerConfig.log(loggerName, fqcn, marker, level, data, t);
    }

    /*
     * (non-Javadoc)
     *
     * @see co.elastic.apm.agent.shaded.apache.logging.log4j.core.config.ReliabilityStrategy#log(co.elastic.apm.agent.shaded.apache.logging.log4j.util.Supplier,
     * java.lang.String, java.lang.String, java.lang.StackTraceElement, co.elastic.apm.agent.shaded.apache.logging.log4j.Marker,
     * co.elastic.apm.agent.shaded.apache.logging.log4j.Level, co.elastic.apm.agent.shaded.apache.logging.log4j.message.Message, java.lang.Throwable)
     */
    @Override
    public void log(final Supplier<LoggerConfig> reconfigured, final String loggerName, final String fqcn,
        final StackTraceElement location, final Marker marker, final Level level, final Message data,
        final Throwable t) {
        loggerConfig.log(loggerName, fqcn, location, marker, level, data, t);
    }

    /*
     * (non-Javadoc)
     *
     * @see co.elastic.apm.agent.shaded.apache.logging.log4j.core.config.ReliabilityStrategy#log(co.elastic.apm.agent.shaded.apache.logging.log4j.util.Supplier,
     * co.elastic.apm.agent.shaded.apache.logging.log4j.core.LogEvent)
     */
    @Override
    public void log(final Supplier<LoggerConfig> reconfigured, final LogEvent event) {
        loggerConfig.log(event);
    }

    /*
     * (non-Javadoc)
     *
     * @see
     * co.elastic.apm.agent.shaded.apache.logging.log4j.core.config.ReliabilityStrategy#beforeLogEvent(co.elastic.apm.agent.shaded.apache.logging.log4j.core.config.
     * LoggerConfig, co.elastic.apm.agent.shaded.apache.logging.log4j.util.Supplier)
     */
    @Override
    public LoggerConfig getActiveLoggerConfig(final Supplier<LoggerConfig> next) {
        return this.loggerConfig;
    }

    /*
     * (non-Javadoc)
     *
     * @see co.elastic.apm.agent.shaded.apache.logging.log4j.core.config.ReliabilityStrategy#afterLogEvent()
     */
    @Override
    public void afterLogEvent() {
        // no action
    }

    /*
     * (non-Javadoc)
     *
     * @see co.elastic.apm.agent.shaded.apache.logging.log4j.core.config.ReliabilityStrategy#beforeStopAppenders()
     */
    @Override
    public void beforeStopAppenders() {
        // no action
    }

    /*
     * (non-Javadoc)
     *
     * @see
     * co.elastic.apm.agent.shaded.apache.logging.log4j.core.config.ReliabilityStrategy#beforeStopConfiguration(co.elastic.apm.agent.shaded.apache.logging.log4j.core
     * .config.Configuration)
     */
    @Override
    public void beforeStopConfiguration(final Configuration configuration) {
        // only sleep once per configuration stop
        if (loggerConfig == configuration.getRootLogger()) {
            try {
                Thread.sleep(SLEEP_MILLIS);
            } catch (final InterruptedException e) {
                StatusLogger.getLogger().warn("Sleep before stop configuration was interrupted.");
            }
        }
    }

}
