/*
 * Copyright (c) 2014-2023 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.security.internal

import android.util.Base64
import java.nio.ByteBuffer
import java.nio.charset.StandardCharsets
import java.security.SecureRandom
import javax.crypto.Cipher
import javax.crypto.SecretKey
import javax.crypto.spec.GCMParameterSpec
import javax.crypto.spec.SecretKeySpec

/**
 * @author Umang Chamaria
 */

private const val GCM_AUTHENTICATION_TAG_LENGTH = 128
private const val GCM_IV_LENGTH = 32
private const val ALGORITHM_NAME_AES = "AES"
private const val CIPHER_TRANSITION = "AES/GCM/NoPadding"

private const val tag = ""

public fun encryptAESGCM(key: ByteArray, plaintext: String): String? {
    val iv = ByteArray(GCM_IV_LENGTH)
    SecureRandom().nextBytes(iv)
    val cipher = Cipher.getInstance(CIPHER_TRANSITION)
    val secretKey: SecretKey = SecretKeySpec(key, ALGORITHM_NAME_AES)
    val parameterSpec = GCMParameterSpec(GCM_AUTHENTICATION_TAG_LENGTH, iv)
    cipher.init(Cipher.ENCRYPT_MODE, secretKey, parameterSpec)
    val cipherText = cipher.doFinal(plaintext.toByteArray())
    val byteBuffer = ByteBuffer.allocate(iv.size + cipherText.size)
    byteBuffer.put(iv)
    byteBuffer.put(cipherText)
    return Base64.encodeToString(byteBuffer.array(), Base64.DEFAULT)
}

public fun decryptAESGCM(key: ByteArray, cipherMessage: String): String? {
    val cipher = Cipher.getInstance(CIPHER_TRANSITION)
    val message = Base64.decode(cipherMessage, Base64.DEFAULT)
    val gcmIv = GCMParameterSpec(GCM_AUTHENTICATION_TAG_LENGTH, message, 0, GCM_IV_LENGTH)
    val secretKey: SecretKey = SecretKeySpec(key, ALGORITHM_NAME_AES)
    cipher.init(Cipher.DECRYPT_MODE, secretKey, gcmIv)
    val plainText = cipher.doFinal(
        message,
        GCM_IV_LENGTH,
        message.size - GCM_IV_LENGTH
    )
    return String(plainText, StandardCharsets.UTF_8)
}