package com.openfin.desktop.channel;

import com.openfin.desktop.Ack;
import com.openfin.desktop.AckListener;
import com.openfin.desktop.channel.webrtc.WebRtcProtocolHandler;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;

public class AbstractProtocolHandler {
    protected Logger logger = LoggerFactory.getLogger(this.getClass().getName());


    protected Channel channel;

    protected boolean isProvider = false;  // created for a channel provider
    protected EndpointIdentity endpointIdentity;  // if isProvider is true, endpoint of the provider
    protected EndpointIdentity clientEndpointIdentity;  // if isProvider is true, client identity of this handler created for
                                                        // if false,  just client identity

    public AbstractProtocolHandler(Channel channel) {
        this.channel = channel;
    }

    public static AbstractProtocolHandler createProtocolHandler(ProtocolOptions options, Channel channel) {
        if (options == Channel.RTC_PROTOCOL) {
            return new WebRtcProtocolHandler(channel);
        } else {
            return new ClassicProtocolHandler(channel);
        }
    }

    public ProtocolOptions getProtocolOptions() {
        return null;
    }

    /**
     * create payload for connecting to provider.  The payload include "offer" object
     *
     * @return connect payload
     * @throws Exception Exception
     */
    public JSONObject getChannelConnectPayload(Object connectPayload) throws Exception {
        this.initializeClient();
        JSONObject payload = new JSONObject();
        payload.put("channelName", this.channel.getName());
        if (Objects.nonNull(connectPayload)) {
            payload.put("payload", connectPayload);
        }
        JSONObject offer = new JSONObject();
        JSONArray protocols = new JSONArray();
        protocols.put(getSupportedOfferProtocol());
        offer.put("supportedProtocols", protocols);
        offer.put("maxProtocols", 2);
        payload.put("offer", offer);
        return payload;
    }

    protected JSONObject getSupportedOfferProtocol() {
        return new JSONObject();
    }

    public void setClientEndpointIdentity(EndpointIdentity endpointIdentity) {
        this.clientEndpointIdentity = endpointIdentity;
    }

    /**
     * Process connection answer response from a channel provider with connect-to-channel message
     *
     * @param res response from a provider
     * @return CompletionStage with true as the answer is accepted
     */
    public CompletionStage<Boolean> processConnectAnswer(JSONObject res) {
        CompletableFuture<Boolean> connectionFuture = new CompletableFuture<>();
        try {
            JSONObject answer = res.optJSONObject("answer");
            JSONObject payload = null;
            if (Objects.nonNull(answer)) {  // older version of Runtime does not send answer
                JSONArray protocols = answer.getJSONArray("supportedProtocols");
                // should be just one protocol in the array,  chosen by the provider
                JSONObject protocol = (JSONObject) protocols.get(0);
                payload = protocol.optJSONObject("payload");
            }
            JSONObject answerObj = Objects.nonNull(payload)?payload.optJSONObject("answer"):null;
            acceptAnswer(answerObj, connectionFuture);
        } catch (Exception ex) {
            logger.error("Error accepting answer", ex);
            connectionFuture.complete(false);
        }
        return connectionFuture;
    }

    protected void acceptAnswer(JSONObject answer, CompletableFuture<Boolean> future) throws Exception {
        future.complete(true);
    }

    public JSONObject getChannelCreatePayload() {
        JSONObject payload = new JSONObject();
        payload.put("channelName", this.channel.getName());
        return payload;
    }

    protected JSONObject getSupportedAnswerProtocol() {
        return new JSONObject();
    }

    /**
     * Process connection offer from a channel client for a provider
     *
     * @param offer offer object from channel client
     * @return answer object if the offer is accepted
     */
    public JSONObject processConnectOffer(JSONObject offer) {
        JSONObject answer = null;
        try {
            JSONObject protocol = getOfferProtocolByType(offer, this.getProtocolOptions());
            JSONObject payload = Objects.nonNull(protocol)?protocol.optJSONObject("payload"):null;
            acceptOffer(payload);
            JSONArray answerProtocols = new JSONArray();
            answerProtocols.put(this.getSupportedAnswerProtocol());
            answer = new JSONObject();
            answer.put("supportedProtocols", answerProtocols);
        } catch (Exception ex) {
            logger.error("Error accepting answer", ex);
        }
        return answer;
    }

    protected void acceptOffer(JSONObject offer) throws Exception {
    }


    public Channel getChannel() { return this.channel; }

    public void initializeClient() throws Exception {
    }

    public void initializeProvider(EndpointIdentity endpointIdentity, EndpointIdentity clientEndpointIdentity) throws Exception {
        this.isProvider = true;
        this.endpointIdentity = endpointIdentity;
        this.clientEndpointIdentity = clientEndpointIdentity;
    }

    public void sendChannelMessage(String action, JSONObject destionationIdentity, JSONObject providerIdentity,
                            JSONObject actionPayload, AckListener ackListener) {
        logger.error("sendChannelMessage not implemented");
    }

    public CompletableFuture<Ack> sendChannelMessageAsync(String action, JSONObject destionationIdentity, JSONObject providerIdentity,
                                                   Object actionPayload) {
        logger.error("sendChannelMessageAsync not implemented");
        return null;
    }

    protected void cleanup() {
        logger.debug("cleanup {} {}", this.channel.getName(), this.clientEndpointIdentity.getEndpointId());
    }

    public static JSONObject getOfferProtocolByType(JSONObject offer, ProtocolOptions options) {
        JSONObject protocol = null;
        if (Objects.nonNull(offer)) {
            JSONArray supportedProtocols = offer.optJSONArray("supportedProtocols");  // protocols supported by the client
            if (Objects.nonNull(supportedProtocols)) {
                // prefer RTC_PROTOCOL over CLASSIC_PROTOCOL if both are specified
                for (int i = 0; i < supportedProtocols.length(); i++) {
                    JSONObject prot = (JSONObject) supportedProtocols.get(i);
                    String type = prot.optString("type");
                    if (Objects.equals(type, options.getName())) {
                        protocol = prot;
                        break;
                    }
                }
            }
        }
        return protocol;
    }

}
