package com.openfin.desktop.channel;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;

import com.openfin.desktop.*;

import com.openfin.desktop.EventListener;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The Channel object allows an OpenFin application to create a channel as a ChannelProvider, or connect to a channel as a ChannelClient.
 * @author Anthony
 *
 */
public class Channel {
	private final static Logger logger = LoggerFactory.getLogger(Channel.class.getName());

	private final DesktopConnection desktopConnection;
	private final String name;

	private ChannelProvider provider;
	private ConcurrentHashMap<String, ChannelClient> clientMap;
	private CopyOnWriteArrayList<ChannelListener> channelListeners;
	private static class ConnectWait {
		public CompletableFuture<ChannelClient> future;
		public AbstractProtocolHandler handler;
		public Object connectPayload;
	}
	private CopyOnWriteArrayList<ConnectWait> pendingConnections;  // pending client connections when waiting for the channel provider to start

	final private static String CONNECTED_EVENT = "connected";
	final private static String DISCONNECTED_EVENT = "disconnected";
	final private static String PROVIDER_NOT_READY_REASON = "internal-nack";  // set when channel provider is not active during connecting
	final public static ProtocolOptions CLASSIC_PROTOCOL = new ProtocolOptions("classic", 1);
	final public static ProtocolOptions RTC_PROTOCOL = new ProtocolOptions("rtc", 1);

	public Channel(String name, DesktopConnection desktopConnection) {
		this.name = name;
		this.desktopConnection = desktopConnection;
		this.clientMap = new ConcurrentHashMap<>();
		this.channelListeners = new CopyOnWriteArrayList<>();
		this.pendingConnections = new CopyOnWriteArrayList<>();
		try {
			this.addEventListener(CONNECTED_EVENT, e->{
				String eventChannelName = e.getEventObject().getString("channelName");
				if (eventChannelName.equals(this.name)) {
					logger.debug("channel[{}] connected", this.name);
					JSONObject providerIdentity = e.getEventObject();
					EndpointIdentity ei = new EndpointIdentity(providerIdentity);
					this.fireChannelConnectEvent(new ConnectionEvent(ei.getChannelId(), ei.getUuid(), ei.getName(), ei.getChannelName(), ei.getEndpointId()));
					//check if any pending connections
					ArrayList<ConnectWait> connectionFutures = new ArrayList<>(this.pendingConnections);
					this.pendingConnections.clear();
					connectionFutures.forEach(connectWait->{
						Channel.this.connectAsyncInternal(null, connectWait.connectPayload, connectWait.handler).thenAccept(client->{
							connectWait.future.complete(client);
						});
					});
				}
			}, null);
			this.addEventListener(DISCONNECTED_EVENT, e->{
				String eventChannelName = e.getEventObject().getString("channelName");
				if (eventChannelName.equals(this.name)) {
					logger.debug("channel[{}] disconnected", this.name);
					JSONObject providerIdentity = e.getEventObject();
					EndpointIdentity ei = new EndpointIdentity(providerIdentity);
					this.fireChannelDisconnectEvent(new ConnectionEvent(ei.getChannelId(), ei.getUuid(), ei.getName(), ei.getChannelName(), ei.getEndpointId()));
				}
			}, null);
		}
		catch (DesktopException e) {
			logger.error("unable to subscribe to channel events", e);
		}
	}

	/**
	 * Get the name of the channel
	 * @return name of the channel
	 */
	public String getName() {
		return this.name;
	}

	public DesktopConnection getDesktopConnection() {
		return this.desktopConnection;
	}

	public CompletionStage<ChannelProvider> createAsync() {
		return this.createAsync(Arrays.asList(Channel.CLASSIC_PROTOCOL));
	}

	public CompletionStage<ChannelProvider> createAsync(List<ProtocolOptions> protocolList) {
		JSONObject payload = new JSONObject();
		payload.put("channelName", this.getName());
		return desktopConnection.sendActionAsync("create-channel", payload, this).thenApply(ack->{
			if (ack.isSuccessful()) {
				JSONObject providerIdentity = (JSONObject) ack.getData();
				EndpointIdentity providerEndpointIdentity = new EndpointIdentity(providerIdentity);
				this.provider =  new ChannelProvider(this, protocolList, providerEndpointIdentity);
				return this.provider;
			}
			else {
				throw new RuntimeException("error creating channel provider, reason: " + ack.getReason());
			}
		});
	}

	public void create(AsyncCallback<ChannelProvider> callback) {
		this.create(callback, CLASSIC_PROTOCOL);
	}

	/**
	 * Create a new channel.
	 * @param callback The callback that receives the wrapped {@link ChannelProvider} object
	 * @param protocolType Type of protocol (CLASSIC or RTC)
	 * @deprecated use {@link #createAsync()} instead
	 */
	public void create(AsyncCallback<ChannelProvider> callback, ProtocolOptions protocolType) {
		List<ProtocolOptions> list = new ArrayList<>();
		list.add(protocolType);
		this.createAsync(list).thenAccept(provider ->{
			callback.onSuccess(provider);
		});
	}

	void destroy(ChannelBase provider, AckListener ackListener) {
		JSONObject payload = new JSONObject();
		payload.put("channelName", provider.getChannelName());
		desktopConnection.sendAction("destroy-channel", payload, new AckListener() {

			@Override
			public void onSuccess(Ack ack) {
				Channel.this.provider = null;
				if (ackListener != null) {
					ackListener.onSuccess(ack);
				}
			}

			@Override
			public void onError(Ack ack) {
				if (ackListener != null) {
					ackListener.onError(ack);
				}
			}
			
		}, this);
	}

	/**
	 * Connect to a channel provider
	 *
	 * @return A new CompletionStage for the channel client
	 */
	public CompletionStage<ChannelClient> connectAsync() {
		return this.connectAsync(null);
	}

	/**
	 * Connect to a channel provider
	 *
	 *  @param connectPayload connection payload
	 * @return A new CompletionStage for the channel client
	 */
	public CompletionStage<ChannelClient> connectAsync(Object connectPayload) {
		return this.connectAsync(connectPayload, CLASSIC_PROTOCOL);
	}

	/**
	 * Connect to a channel provider
	 *
	 * @param connectPayload connection payload
	 * @param protocolType requested protocol type for the channel
	 * @return A new CompletionStage for the channel client
	 */
	public CompletionStage<ChannelClient> connectAsync(Object connectPayload, ProtocolOptions protocolType) {
		return this.connectAsync(null, connectPayload, protocolType);
	}

	/**
	 * Connect to a channel provider
	 *
	 * @param noWait true if the request does not wait when the channel provider is not active
	 *
	 * @param connectPayload connection payload
	 * @param protocolType requested protocol type for the channel
	 * @return A new CompletionStage for the channel client
	 */
	public CompletionStage<ChannelClient> connectAsync(Boolean noWait, Object connectPayload, ProtocolOptions protocolType) {
		AbstractProtocolHandler protocolHandler = AbstractProtocolHandler.createProtocolHandler(protocolType, this);
		return connectAsyncInternal(noWait, connectPayload, protocolHandler);
	}

	private CompletionStage<ChannelClient> connectAsyncInternal(Boolean noWait, Object connectPayload, AbstractProtocolHandler protocolHandler) {
		CompletableFuture<ChannelClient> connectionFuture = new CompletableFuture<>();
		try {
			desktopConnection.sendActionAsync("connect-to-channel", protocolHandler.getChannelConnectPayload(connectPayload), this).thenAccept(ack -> {
				if (ack.isSuccessful()) {
					JSONObject res = (JSONObject) ack.getData();
					protocolHandler.processConnectAnswer(res).thenAccept(connected -> {
						if (connected) {
							EndpointIdentity endpointIdentity = new EndpointIdentity(res);
							ChannelClient client = new ChannelClient(protocolHandler, endpointIdentity);
							this.clientMap.put(endpointIdentity.getEndpointId(), client);
							protocolHandler.setClientEndpointIdentity(endpointIdentity);
							connectionFuture.complete(client);
						} else {
							connectionFuture.completeExceptionally(new RuntimeException("error connecting to channel " + this.name + ", reason: failed to accept answer from provider"));
						}
					});
				} else {
					if (!PROVIDER_NOT_READY_REASON.equals(ack.getReason())) {
						connectionFuture.completeExceptionally(new RuntimeException("error connecting to channel " + this.name + ", reason: " + ack.getReason()));
					} else if (noWait != null && noWait.booleanValue()) {
						connectionFuture.completeExceptionally(new RuntimeException("error connecting to channel " + this.name + ", reason: " + ack.getReason()));
					} else {
						//connect to the provider when it's ready.
						ConnectWait wait = new ConnectWait();
						wait.future = connectionFuture;
						wait.handler = protocolHandler;
						wait.connectPayload = connectPayload;
						this.pendingConnections.add(wait);
					}
				}
			});
		} catch (Exception ex) {
			logger.error("Error connecting to channel", ex);
		}
		return connectionFuture;
	}
	
	/**
	 * Connect to the channel.
	 * @param callback The callback that receives the wrapped {@link ChannelClient} object
	 * @deprecated use {@link #connectAsync(Object)} instead
	 */
	public void connect(AsyncCallback<ChannelClient> callback) {
		this.connect(CLASSIC_PROTOCOL, callback);
	}

	/**
	 * Connect to the channel.
	 * @param callback The callback that receives the wrapped {@link ChannelClient} object
	 * @param protocolType Type of protocol ("classic" or "rtc")
	 * @deprecated use {@link #connectAsync(Object, ProtocolOptions)}  } instead
	 */
	public void connect(ProtocolOptions protocolType, AsyncCallback<ChannelClient> callback) {
		this.connectAsync(null, protocolType).thenAccept(client->{
			callback.onSuccess(client);
		});
	}

	CompletableFuture<Ack> disconnect(ChannelBase client) {
		JSONObject payload = new JSONObject();
		payload.put("channelName", client.getChannelName());
		payload.put("uuid", client.getUuid());
		payload.put("name", client.getName());
		payload.put("endpointId", client.getEndpointId());
		return desktopConnection.sendActionAsync("disconnect-from-channel", payload, this);
	}

	/**
	 * Check if the channel has provider created
	 * @return if the channel provider exists.
	 */
	public boolean hasProvider() {
		return this.provider != null;
	}

	/**
	 * Check if the channel has client connected
	 * @return true if a channel client exists
	 */
	public boolean hasClient() {
		return this.clientMap.keySet().size() > 0;
	}

	public Object invokeAction(EndpointIdentity targetIdentity, String action, Object actionPayload,
			JSONObject senderIdentity) {
		Object result = null;
		if (targetIdentity.getEndpointId() != null && this.clientMap.contains(targetIdentity.getEndpointId())) { //specified ChannelClient.
			//target should be channel client.
			ChannelClient channelClient = this.clientMap.get(targetIdentity.getEndpointId());
			if (channelClient != null && channelClient.hasRegisteredAction(action)) {
				result = channelClient.invokeAction(action, actionPayload, senderIdentity);
			}
		}
		else { 
			if (this.provider != null && this.provider.hasRegisteredAction(action)) {
				result = provider.invokeAction(action, actionPayload, senderIdentity);
			}
			else {
				Collection<ChannelClient> clients = this.clientMap.values();
				JSONArray results = new JSONArray();
				for (ChannelClient client :  clients) {
					if (client.hasRegisteredAction(action)) {
						Object cr = client.invokeAction(action, actionPayload, senderIdentity);
						if (cr != null) {
							results.put(cr);
						}
					}
				}
				result = results.length() == 1 ? results.get(0) : results;
			}
		}
		return result;
	}

	public boolean addChannelListener(ChannelListener listener) {
		return this.channelListeners.add(listener);
	}

	public boolean removeChannelListener(ChannelListener listener) {
		return this.channelListeners.remove(listener);
	}

	protected void fireChannelConnectEvent(ConnectionEvent event) {
		for (ChannelListener listener : this.channelListeners) {
			listener.onChannelConnect(event);
		}
		this.clientMap.values().forEach(channelClient->{
			channelClient.fireChannelConnectEvent(event);
		});
	}

	protected void fireChannelDisconnectEvent(ConnectionEvent event) {
		for (ChannelListener listener : this.channelListeners) {
			listener.onChannelDisconnect(event);
		}
		this.clientMap.values().forEach(channelClient->{
			channelClient.fireChannelDisconnectEvent(event);
		});
	}

	public void processConnection(JSONObject payload, JSONObject ackToSenderPayload) throws Exception {
		this.provider.processConnection(payload, ackToSenderPayload);
	}

	/**
	 * Registers an event listener on the specified event
	 * <pre>
	 *     Supported system event types are:
	 * </pre>
	 *
	 * @param subscriptionObject A JSON object containing subscription information such as the topic and type
	 * @param listener EventListener for the event
	 * @param callback AckListener for the request
	 * @throws DesktopException if this method fails to add event listener specified
	 * @see EventListener
	 * @see AckListener
	 */
	protected void addEventListener(JSONObject subscriptionObject,
									EventListener listener,
									AckListener callback) throws DesktopException {
		this.desktopConnection.addEventCallback(subscriptionObject, listener, callback, this);
	}

	/**
	 * Registers an event listener on the specified event
	 * <pre>
	 *     Supported system event types are:
	 * </pre>
	 *
	 * @param type Type of the event
	 * @param listener EventListener for the event
	 * @param callback AckListener for the request
	 * @throws  DesktopException if this method fails to add event listener specified
	 * @see EventListener
	 * @see AckListener
	 */
	public void addEventListener(String type, EventListener listener, AckListener callback) throws DesktopException {
		try {
			JSONObject eventListenerPayload = new JSONObject();
			eventListenerPayload.put("topic", "channel");
			eventListenerPayload.put("type", type);
			addEventListener(eventListenerPayload, listener, callback);
		} catch (Exception e) {
			logger.error("Error adding event listener", e);
			throw new DesktopException(e);
		}
	}

	/**
	 * Removes a previously registered event listener from the specified event
	 *
	 * @param type Type of the event
	 * @param listener EventListener for the event
	 * @param callback AckListener for the request
	 * @throws  DesktopException if this method fails to remove event listener specified
	 * @see DesktopException
	 * @see EventListener
	 * @see AckListener
	 */
	public void removeEventListener(String type, EventListener listener, AckListener callback) throws DesktopException{
		try {
			JSONObject eventListenerPayload = new JSONObject();
			eventListenerPayload.put("topic", "channel");
			eventListenerPayload.put("type", type);
			this.desktopConnection.removeEventCallback(eventListenerPayload, listener, callback, this);
		} catch (Exception e) {
			logger.error("Error removing event listener", e);
			throw new DesktopException(e);
		}
	}

	/**
	 * Raise a channel event to Runtime
	 *
	 * @param type type of event
	 * @param eventObject event payload
	 * @throws DesktopException DesktopException
	 * @return CompletableFuture with Ack object
	 * @see Ack
	 */
	public CompletableFuture<Ack> raiseEvent(String type, JSONObject eventObject) throws DesktopException {
		JSONObject payload = new JSONObject();
		payload.put("eventName", String.format("channel/%s", type));
		payload.put("eventArgs", eventObject);
		return this.desktopConnection.sendActionAsync("raise-event", payload, this);
	}

}
