package com.openfin.desktop.channel;

import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;

import org.json.JSONObject;

import com.openfin.desktop.Ack;
import com.openfin.desktop.AckListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ChannelBase {

	protected Middleware defaultAction;
	protected Middleware onError;
	protected Middleware beforeAction;
	protected Middleware afterAction;
	protected ConcurrentHashMap<String, ChannelAction> channelActionMap;
	protected EndpointIdentity endpointIdentity;

	protected Logger logger = LoggerFactory.getLogger(this.getClass().getName());

	public ChannelBase(EndpointIdentity endpointIdentity) {
		this.endpointIdentity = endpointIdentity;
		this.channelActionMap = new ConcurrentHashMap<String, ChannelAction>();
	}

	protected boolean register(String action, ChannelAction listener) {
		return this.channelActionMap.putIfAbsent(action, listener) == null;
	}

	/**
	 * remove the registered action
	 * @param action Name of the action to be removed.
	 * @return if the action was removed successfully
	 */
	public boolean remove(String action) {
		return this.channelActionMap.remove(action) != null;
	}

	/**
	 * Sets a default action. This is used any time an action that has not been registered is invoked.
	 * @param middleware Action to be executed when a client invokes an action name that has not been registered.
	 */
	public void setDefaultAction(Middleware middleware) {
		this.defaultAction = middleware;
	}

	/**
	 * Register an error handler. This is called before responding on any error.
	 * @param middleware Action to be executed in case of an error.
	 */
	public void setOnError(Middleware middleware) {
		this.onError = middleware;
	}

	/**
	 * Register an action that fires before the action.
	 * @param middleware Action to be executed before invoking the action.
	 */
	public void setBeforeAction(Middleware middleware) {
		this.beforeAction = middleware;
	}

	/**
	 * Register an action that fires after the action.
	 * @param middleware  Action to be executed after invoking the action.
	 */
	public void setAfterAction(Middleware middleware) {
		this.afterAction = middleware;
	}
	
	public EndpointIdentity getEndpointIdentity() {
		return this.endpointIdentity;
	}

	public String getChannelName() {
		return this.endpointIdentity.getChannelName();
	}

	public String getUuid() {
		return this.endpointIdentity.getUuid();
	}

	public String getChannelId() {
		return this.endpointIdentity.getChannelId();
	}

	public String getName() {
		return this.endpointIdentity.getName();
	}
	
	public String getEndpointId() {
		return this.endpointIdentity.getEndpointId();
	}

	public boolean hasRegisteredAction(String action) {
		return channelActionMap.containsKey(action);
	}

	/**
	 * Dispatch an action to a specified client.
	 *
	 * @param protocolHandler protocol handler
	 * @param destinationIdentity Identity of the target client.
	 * @param action Name of the action to be invoked by the client.
	 * @param actionPayload Payload to be sent along with the action.
	 * @param ackListener AckListener for the request
	 */
	protected void dispatch(AbstractProtocolHandler protocolHandler, JSONObject destinationIdentity, String action, JSONObject actionPayload,
			AckListener ackListener) {
		protocolHandler.sendChannelMessage(action, destinationIdentity, this.endpointIdentity.toJSON(), actionPayload, ackListener);
	}

	protected CompletableFuture<Ack> dispatchAsync(AbstractProtocolHandler protocolHandler, JSONObject destinationIdentity, String action, Object actionPayload) {
		return protocolHandler.sendChannelMessageAsync(action, destinationIdentity, this.endpointIdentity.toJSON(), actionPayload);
	}

	Object invokeAction(String action, Object payload, JSONObject senderIdentity) {
		try {
			ChannelAction channelAction = this.channelActionMap.get(action);
			if (channelAction != null || this.defaultAction != null ) {
				if (this.beforeAction != null) {
					payload = this.beforeAction.invoke(action, payload, senderIdentity);
				}
				if (channelAction != null) {
					payload = channelAction.invoke(action, payload, senderIdentity);
				}
				else {
					payload = this.defaultAction.invoke(action, payload, senderIdentity);
				}
				
				if (this.afterAction != null) {
					payload = this.afterAction.invoke(action, payload, senderIdentity);
				}
				return payload;
			}
			else {
				//error, throw exception?
				return null;
			}
		}
		catch(Exception ex) {
			if (this.onError != null) {
				this.onError.invoke(action, payload, senderIdentity);
			}
			ex.printStackTrace();
			return null;
		}
		finally {
		}
	}

	protected void cleanup() {
		logger.debug("cleaning up {} {}", this.getChannelName(), this.getEndpointId());
		this.channelActionMap.clear();
//		if (Objects.nonNull(this.protocolHandler)) {
//			this.protocolHandler.cleanup();
//			this.protocolHandler = null;
//		}
	}
}
